import datetime
from django.utils import timezone
from django.utils.encoding import force_text
from rest_framework import generics, exceptions, response, status

from intranet.crt.api.v1.certificates.serializers.dispatch import CertificateSerializerDispatcher
from intranet.crt.api.base.serializer_mixins.reissue import (
    PcReissueValidator, RcServerReissueValidator, YcServerReissueValidator, VpnTokenReissueValidator,
    ClientServerReissueValidator, BaseReissueValidator, TpmSmartcard1CReissueValidator,
    BankClientServerReissueValidator,
)

from intranet.crt.constants import CERT_TYPE, CERT_STATUS
from intranet.crt.core.ca.exceptions import ValidationCaError
from intranet.crt.api.base.exceptions import BadRequest
from intranet.crt.core.models import Certificate


class ReissueCertificateView(generics.GenericAPIView):
    permission_classes = []
    serializer_class = CertificateSerializerDispatcher

    def __init__(self, **kwargs):
        super(ReissueCertificateView, self).__init__(**kwargs)
        self.validators = {
            CERT_TYPE.PC: PcReissueValidator,
            CERT_TYPE.BANK_PC: PcReissueValidator,
            CERT_TYPE.ZOMB_PC: PcReissueValidator,
            CERT_TYPE.RC_SERVER: RcServerReissueValidator,
            CERT_TYPE.YC_SERVER: YcServerReissueValidator,
            CERT_TYPE.VPN_TOKEN: VpnTokenReissueValidator,
            CERT_TYPE.CLIENT_SERVER: ClientServerReissueValidator,
            CERT_TYPE.BANK_CLIENT_SERVER: BankClientServerReissueValidator,
            CERT_TYPE.TPM_SMARTCARD_1C: TpmSmartcard1CReissueValidator,
        }

    def get_validator_class(self, cert_type_name):
        return self.validators.get(cert_type_name, BaseReissueValidator)

    def get_old_certificate(self):
        serial_number = self.request.environ.get('HTTP_X_SSL_CLIENT_SERIAL')
        if self.request.environ.get('HTTP_X_SSL_CLIENT_VERIFY') == 'NOT_KNOWN_TO_CRT':
            raise BadRequest(
                'Client certificate with given serial number not found ({})'.format(serial_number)
            )
        elif self.request.environ.get('HTTP_X_SSL_CLIENT_VERIFY') != '0':
            raise BadRequest(
                'Client certificate is not provided. Please provide client certificate for reissue'
            )

        try:
            old_cert = Certificate.objects.get(
                serial_number=serial_number,
                status=CERT_STATUS.ISSUED,
            )
        except Certificate.DoesNotExist:
            raise exceptions.ParseError(
                'Certificate with the serial number of the client certificate ({}) was not found'
                .format(serial_number)
            )

        return old_cert

    def post(self, *args, **kwargs):
        old_cert = self.get_old_certificate()
        validator_class = self.get_validator_class(old_cert.type.name)
        validator = validator_class(self.request)

        cert_user = old_cert.get_reissue_user()

        additional_fields = validator.get_additional_fields(old_cert)
        if 'requested_by_csr' not in additional_fields:
            additional_fields['requested_by_csr'] = 'request' in self.request.data

        new_cert = Certificate.objects.create(
            type=old_cert.type,
            ca_name=old_cert.ca_name,
            email=old_cert.email,
            requester=cert_user,
            user=cert_user,
            common_name=old_cert.common_name,
            is_reissue=True,
            desired_ttl_days=validator.get_desired_desired_ttl_days(old_cert),
            **additional_fields
        )

        try:
            new_cert.controller.issue()
        except ValidationCaError as e:
            raise BadRequest(force_text(e))

        if new_cert.status != CERT_STATUS.ISSUED:
            raise exceptions.APIException(
                'Certificate was not reissued: %s' % new_cert.error_message, code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

        revoke_at = timezone.now() + datetime.timedelta(days=validator.old_cert_revoke_days)
        old_cert.controller.add_to_hold_queue(revoke_at, description='reissued')

        for same_pc_inum_cert in new_cert.get_same_pc_inum_certs().issued().exclude(pk=old_cert.pk):
            same_pc_inum_cert.controller.add_to_hold_queue(
                revoke_at, description='reissued with same pc_inum by {}'.format(new_cert.pk)
            )

        serializer = self.get_serializer(new_cert)
        data = serializer.data
        data['certificate'] = new_cert.certificate
        return response.Response(data, status=status.HTTP_201_CREATED)
