from rest_framework import generics
from rest_framework_xml.renderers import XMLRenderer

from intranet.crt.api.base.views import NoAuthenticationMixin
from intranet.crt.core.models import HostToApprove, HostToApproveHistory, HostValidationCode
from intranet.crt.constants import HOST_VALIDATION_CODE_STATUS
from .serializers import HostHistorySerializer, HostToApproveDnsSerializer, ManagedHostSerializer, HostValidationCodeSerializer


class HostsToApproveList(NoAuthenticationMixin, generics.ListAPIView):
    model = HostValidationCode
    pagination_class = None
    serializer_class = HostValidationCodeSerializer

    def get_serializer(self, instance=None, data=None, many=False,
                       partial=False, allow_add_remove=False):

        if self.request.method == 'GET':
            selected_fields = self.request.GET.get('fields')
            if selected_fields:
                if selected_fields == '__all__':
                    selected_fields = None
                else:
                    selected_fields = [field.strip()
                                       for field in selected_fields.split(',')]
            else:
                selected_fields = self.serializer_class.Meta.fields

            class Serializer(self.serializer_class):
                class Meta(self.serializer_class.Meta):
                    fields = selected_fields

        else:
            Serializer = self.serializer_class

        if data:
            return Serializer(instance, data=data, many=many, partial=partial)
        else:
            return Serializer(instance, many=many, partial=partial)

    def get_queryset(self):
        only_actual = self.request.GET.get('all') is None
        queryset = HostValidationCode.objects.all()

        if only_actual:
            queryset = queryset.filter(
                status=HOST_VALIDATION_CODE_STATUS.validation,
            ).select_related('host')
            queryset = [code for code in queryset if code.is_waiting_for_validation()]

        return queryset


class HostToApproveDetail(NoAuthenticationMixin, generics.RetrieveAPIView):
    model = HostToApprove
    queryset = HostToApprove.objects.all()
    pagination_class = None
    serializer_class = HostToApproveDnsSerializer

    def get_serializer(self, instance=None, data=None, many=False,
                       partial=False, allow_add_remove=False):

        if self.request.method == 'GET':
            selected_fields = self.request.GET.get('fields')
            if selected_fields:
                if selected_fields == '__all__':
                    selected_fields = None
                else:
                    selected_fields = [field.strip()
                                       for field in selected_fields.split(',')]
            else:
                selected_fields = self.serializer_class.Meta.fields

            class Serializer(self.serializer_class):
                class Meta(self.serializer_class.Meta):
                    fields = selected_fields

        else:
            Serializer = self.serializer_class

        if data:
            return Serializer(instance, data=data, many=many, partial=partial)
        else:
            return Serializer(instance, many=many, partial=partial)


class HostToApproveHistoryList(NoAuthenticationMixin, generics.ListAPIView):
    model = HostToApproveHistory
    pagination_class = None
    serializer_class = HostHistorySerializer

    def get_queryset(self):
        queryset = super(HostToApproveHistoryList, self).get_queryset()
        pk = self.kwargs['pk']
        queryset = queryset.filter(host__pk=pk).order_by('-created')
        return queryset


class HostsToApproveList2(HostsToApproveList):
    renderer_classes = (XMLRenderer,)


class AutoManagedHostsList(NoAuthenticationMixin, generics.ListAPIView):
    model = HostToApprove
    pagination_class = None
    # позволяем просматривать этот список всем подряд
    permission_classes = []
    serializer_class = ManagedHostSerializer

    def get_queryset(self):
        return HostToApprove.objects.filter(auto_managed=True)

    def filter_queryset(self, queryset):
        if 'managed_dns' in self.request.GET:
            FALSY_VALUES = ['0', 'false']
            managed_dns = self.request.GET['managed_dns'].strip().lower() not in FALSY_VALUES
            return queryset.filter(managed_dns=managed_dns)
        return queryset
