import itertools
from rest_framework import permissions

from intranet.crt.api.v1.certificates.permissions import IsOwner
from intranet.crt.constants import CERT_TYPE


class CanViewCertificate(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        if request.user.has_perm('core.can_view_any_certificate'):
            return True

        return IsOwner().has_object_permission(request, view, obj)


class CanDownloadCertificate(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        if request.user.has_perm('core.can_download_any_certificate'):
            return True

        return IsOwner().has_object_permission(request, view, obj)


class CanRevokeCertificate(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        if request.user.has_perm('core.can_revoke_any_certificate'):
            return True
        if request.user.has_perm('core.can_revoke_users_certificates') and obj.type.name in CERT_TYPE.USERS_TYPES:
            return True
        return IsOwner().has_object_permission(request, view, obj)


class CanUpdateCertificateData(permissions.BasePermission):
    """
    Этот пермишн проверяется при каждом get-запросе,
    и в таком случае мы хотим автоматически возвращать True.
    В то же время иногда необходимо явно узнать,
    может ли пользователь изменять данные сертификата.
    Для этого используется параметр simulate_strict=True.
    """
    def has_object_permission(self, request, view, obj, simulate_strict=False):
        if not simulate_strict and request.method in permissions.SAFE_METHODS:
            return True

        is_owner = IsOwner().has_object_permission(request, view, obj)

        # Проверка на SAFE_METHODS здесь – костыль,
        # чтобы в GET-запросе (список пермишнов в API, действия на фронте)
        # учитывалась только возможность редактировать abc_service
        if 'abc_service' in request.POST or request.method in permissions.SAFE_METHODS:
            if not request.user.has_perm('core.can_change_abc_service_for_any_certificate') and not is_owner:
                return False
        if 'manual_tags' in request.POST:
            if not request.user.has_perm('core.can_edit_certificate_tags'):
                return False

        return True


ACTIONS_INFO = {
    'update': [CanViewCertificate, CanUpdateCertificateData],
    'download': [CanViewCertificate, CanDownloadCertificate],
    'hold': [CanViewCertificate, CanRevokeCertificate],
    'unhold': [CanViewCertificate, CanRevokeCertificate],
    'revoke': [CanViewCertificate, CanRevokeCertificate],
}


def gather_permissions(actions):
    all_permissions = itertools.chain(*(ACTIONS_INFO[action] for action in actions))
    return sorted(set(all_permissions), key=lambda x: x.__name__)
