from rest_framework import exceptions, relations, serializers

from intranet.crt.api.v1.certificates.fields import DownloadField
from intranet.crt.api.v1.certificates.serializers import base
from intranet.crt.api.v1.certificates.serializers.base import BaseCertSerializer, exclude
from intranet.crt.api.v2.certificates.permissions import CanUpdateCertificateData, ACTIONS_INFO
from intranet.crt.api.v2.misc.fields import (UserField,
                                    CertStatusField,
                                    CertTypeField,
                                    ExtendableFieldMixin,
                                    CaNameField)

from intranet.crt.constants import CERT_ACTION
from intranet.crt.core.models import Certificate, ApproveRequest, CertificateType
from intranet.crt.users.models import CrtUser

COMMON_READONLY_FIELDS = exclude(base.COMMON_READONLY_FIELDS, 'username', 'download2') + ('user', 'id')
EXPLICIT_READONLY_FIELDS = base.EXPLICIT_READONLY_FIELDS


class BaseMeta:
    model = Certificate
    read_only_fields = COMMON_READONLY_FIELDS
    fields = COMMON_READONLY_FIELDS + EXPLICIT_READONLY_FIELDS


class BaseV2CertSerializer(BaseCertSerializer):
    class Meta(BaseMeta):
        pass

    username = None  # Это было полем
    user = UserField(
        queryset=CrtUser.objects.all(),
        slug_field='username',
        required=False,
    )
    requester = UserField(
        queryset=CrtUser.objects.all(),
        slug_field='username',
        required=False,
    )
    ca_name = CaNameField(required=True)
    type = CertTypeField(
        queryset=CertificateType.objects.filter(is_active=True),
        slug_field='name',
    )
    status = CertStatusField(required=False)
    url = relations.HyperlinkedIdentityField(
        view_name='v2:certificate',
        lookup_field='pk',
        read_only=True,
    )
    download = DownloadField(view_name='v2:certificate-download', format='pfx')  # download2 в v1
    download2 = None

    def __init__(self, *args, **kwargs):
        self.api_options = kwargs.pop('api_options', None)
        super(BaseV2CertSerializer, self).__init__(*args, **kwargs)

    def stringify_nulls(self, data):
        if not self.api_options.stringify_nulls:
            return

        stringified_types = (
            serializers.CharField,
            serializers.DateTimeField,
        )
        stringified_fields = (  # для особо хитровыделанных полей
            'st_issue_key',
        )

        for key, value in data.items():
            if value is not None:
                continue
            if key in stringified_fields:
                data[key] = ''
                continue
            for type_ in stringified_types:
                if isinstance(self.fields[key], type_):
                    data[key] = ''
                    break

    def extend_fields(self, data):
        for name, value in list(data.items()):
            field = self.fields[name]
            if isinstance(field, ExtendableFieldMixin):
                data[name + '_human'] = value['human']
                data[name] = value['value']

    def exclude_fields(self, data):
        params = self.context['request'].query_params
        raw_fields = params.get('_fields')

        # Ограничиваем выдачу полей только в списковой ручке
        if not isinstance(self.parent, serializers.ListSerializer):
            return

        if not raw_fields:
            raise exceptions.ParseError('_fields parameter is required')
        fields = set(raw_fields.split(','))

        fields.add('id')
        for field in dict(data):
            if field not in fields:
                del data[field]

    def add_permissions(self, data, instance):
        request = self.context['request']

        permitted_actions = []
        for action, permissions in ACTIONS_INFO.items():
            can_access = True
            for perm in permissions:
                args = [request, None, instance]  # None - view, который в наших пермишнах не используется
                if perm == CanUpdateCertificateData:
                    args.append(True)  # simulate_strict

                try:
                    result = perm().has_object_permission(*args)
                except exceptions.PermissionDenied:
                    result = False
                if not result:
                    can_access = False
                    break
            if can_access:
                permitted_actions.append(action)

        available_actions = [
            action for action in permitted_actions
            if action in CERT_ACTION.available_by_status[instance.status]
        ]
        data['available_actions'] = [CERT_ACTION.humanize(action) for action in available_actions]

    def to_representation(self, instance):
        result = super(BaseCertSerializer, self).to_representation(instance)

        self.stringify_nulls(result)
        self.extend_fields(result)
        self.exclude_fields(result)

        # В списковой ручке не выводим
        if not isinstance(self.parent, serializers.ListSerializer):
            self.add_permissions(result, instance)
        return result
