from django.utils.translation import ugettext_lazy as _
import re
from rest_framework import serializers, relations

from intranet.crt.api.base.serializer_mixins.botik import BotikCertMixin
from intranet.crt.api.base.serializer_mixins.clientserver import ClientServerMixin, BankClientServerMixin
from intranet.crt.api.base.serializer_mixins.host import HostsCertMixin, ExternalHostsCertMixin
from intranet.crt.api.base.serializer_mixins.imdm import ImdmCertMixin
from intranet.crt.api.base.serializer_mixins.personal import (
    SelfRequestOnlyMixin, NinjaCertMixin, HypercubeCertMixin, PcCertMixin, BankPcCertMixin,
    AssessorCertMixin, ZombPcCertMixin, TempPcCertMixin, NinjaExchangeCertMixin,
)
from intranet.crt.api.base.serializer_mixins.mobvpn import MobvpnCertMixin
from intranet.crt.api.base.serializer_mixins.rcserver import RcServerCertMixin
from intranet.crt.api.base.serializer_mixins.vpntoken import VpnTokenCertMixin
from intranet.crt.api.base.serializer_mixins.zombie import ZombieCertMixin
from intranet.crt.api.base.serializer_mixins.ycserver import YcServerCertMixin
from intranet.crt.api.base.serializer_mixins.mdb import MdbCertMixin
from intranet.crt.api.base.serializer_mixins.postamate import PostamateCertMixin
from intranet.crt.api.base.serializer_mixins.sdc import SdcCertMixin
from intranet.crt.api.base.serializer_mixins.tpm_smartcard_1c import TpmSmartcard1CCertMixin
from intranet.crt.api.base.serializer_mixins.vpn_1d import Vpn1DCertMixin
from intranet.crt.api.v2.certificates.serializers.base import BaseV2CertSerializer, exclude, BaseMeta
from intranet.crt.api.v2.misc.fields import AbcServiceField
from intranet.crt.api.v1.hosttoapprove.serializers import HostToApproveSerializer

from intranet.crt.core.models import ApproveRequest, Host
from intranet.crt.utils.punicode import decode_punicode


class BotikCertSerializer(BotikCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        required = ('common_name',)
        fields = BaseMeta.fields + ('helpdesk_ticket',)


class ClientServerCertSerializer(ClientServerMixin, BaseV2CertSerializer):
    approve_request = serializers.SlugRelatedField(
        queryset=ApproveRequest.objects.all(),
        slug_field='id',
        required=False,
    )
    abc_service = AbcServiceField(
        slug_field='external_id',
        required=False,
        style={'base_template': 'input.html'}  # чтобы не тупило и не проставляло дефолт
    )

    class Meta(BaseMeta):
        new_readonly = ('approve_request',)
        new_readwrite = ('abc_service', 'request')
        read_only_fields = BaseMeta.read_only_fields + new_readonly
        fields = BaseMeta.fields + new_readonly + new_readwrite
        required = ('request',)


class BankClientServerCertSerializer(BankClientServerMixin, BaseV2CertSerializer):
    approve_request = serializers.SlugRelatedField(
        queryset=ApproveRequest.objects.all(),
        slug_field='id',
        required=False,
    )
    abc_service = AbcServiceField(
        slug_field='external_id',
        required=False,
        style={'base_template': 'input.html'}  # чтобы не тупило и не проставляло дефолт
    )

    class Meta(BaseMeta):
        new_readonly = ('approve_request',)
        new_readwrite = ('abc_service', 'request')
        read_only_fields = BaseMeta.read_only_fields + new_readonly
        fields = BaseMeta.fields + new_readonly + new_readwrite
        required = ('request',)


class PostamateCertSerializer(PostamateCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        required = ('common_name',)


class SdcCertSerializer(SdcCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        required = ('common_name',)


class HostListSerializer(serializers.ListSerializer):
    split_with_whitespaces_re = re.compile(r',|\s')

    def parse_hosts_field(self, value):
        return [{'hostname': host.strip()} for host in value.split(',') if host.strip()]

    def get_value(self, dictionary):
        return dictionary.get(self.field_name, serializers.empty)

    def to_internal_value(self, value):
        hosts = self.parse_hosts_field(value) if value else []
        return super(HostListSerializer, self).to_internal_value(hosts)

    class Meta:
        model = Host
        fields = ('hostname',)


class HostSerializer(serializers.ModelSerializer):
    host_re = re.compile(r'^(\*\.)?((\w[\w-]{1,62}\w|\w\w?)\.)+(\w[\w-]{1,62}\w|\w\w?)$', re.UNICODE)

    def run_validators(self, value):
        # Внутри базовой функции происходит что-то страшное, но не нужное
        pass

    def to_internal_value(self, data):
        data['hostname'] = decode_punicode(data['hostname'])
        self.check_host(data['hostname'])
        return Host.objects.get_or_create(hostname=data['hostname'])[0]

    def check_host(self, hostname):
        error_message = _(
            'Host \'%s\' contains invalid formatting, e.g. '
            'contains two or more asterisks, '
            'non-leading asterisks, whitespace characters, etc.'
        ) % hostname
        if '_' in hostname or len(hostname) > 255 or not self.host_re.match(hostname):
            exc = serializers.ValidationError(error_message)
            exc.detail = exc.detail[0]  # Получать всегда список из одного элемента - не по-пацански
            raise exc

    class Meta:
        model = Host
        fields = ('hostname',)
        list_serializer_class = HostListSerializer


class HostsCertSerializer(HostsCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        new_readonly = ('email', 'approve_request', 'st_issue_key', 'hosts_to_approve')
        new_readwrite = ('hosts', 'extended_validation', 'request', 'abc_service', 'notify_on_expiration')

        read_only_fields = exclude(BaseMeta.read_only_fields, 'is_ecc', 'common_name') + new_readonly
        fields = BaseMeta.fields + new_readonly + new_readwrite

    request = serializers.CharField(required=False, trim_whitespace=False)
    hosts = HostSerializer(required=False, many=True)
    hosts_to_approve = HostToApproveSerializer(many=True, read_only=True)
    approve_request = serializers.SlugRelatedField(
        queryset=ApproveRequest.objects.all(),
        slug_field='id',
        required=False,
    )
    st_issue_key = serializers.SlugRelatedField(
        queryset=ApproveRequest.objects.all(),
        slug_field='st_issue_key',
        required=False,
        source='approve_request',
    )
    abc_service = AbcServiceField(slug_field='external_id', required=True)
    notify_on_expiration = serializers.BooleanField(default=True)


class ExternalHostsCertSerializer(ExternalHostsCertMixin, HostsCertSerializer):
    pass


class ImdmCertSerializer(ImdmCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'user')
        fields = BaseMeta.fields + ('request', 'pc_serial_number', 'helpdesk_ticket')
        required = ('request', 'pc_serial_number')

    request = serializers.CharField(required=True, trim_whitespace=False)
    pc_serial_number = serializers.CharField(required=True)


class NinjaCertSerializer(NinjaCertMixin, BaseV2CertSerializer):
    class Meta(BaseV2CertSerializer.Meta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')


class NinjaExchangeCertSerializer(NinjaExchangeCertMixin, BaseV2CertSerializer):
    class Meta(BaseV2CertSerializer.Meta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')


class HypercubeCertSerializer(HypercubeCertMixin, BaseV2CertSerializer):
    class Meta(BaseV2CertSerializer.Meta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        required = ('common_name', 'desired_ttl_days')


class TempPcCertSerializer(TempPcCertMixin, BaseV2CertSerializer):
    request = serializers.CharField(required=True, trim_whitespace=False)

    class Meta(BaseV2CertSerializer.Meta):
        fields = BaseMeta.fields + ('request', 'helpdesk_ticket',)


class PcCertSerializer(PcCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        fields = BaseMeta.fields + (
            'pc_hostname', 'pc_os', 'pc_serial_number', 'pc_mac', 'pc_inum', 'request', 'helpdesk_ticket'
        )
        required = ('pc_os', 'pc_hostname', 'pc_mac', 'request')
        optional = ('pc_inum', 'pc_serial_number', 'helpdesk_ticket')

    request = serializers.CharField(required=False, trim_whitespace=False)


class BankPcCertSerializer(BankPcCertMixin, PcCertSerializer):
    class Meta(PcCertSerializer.Meta):
        pass


class ZombPcCertSerializer(ZombPcCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        fields = PcCertSerializer.Meta.fields + ('hardware_request_st_id',)
        required = PcCertSerializer.Meta.required + ('hardware_request_st_id',)
        optional = PcCertSerializer.Meta.optional


class LinuxPcCertSerializer(PcCertSerializer):
    class Meta(PcCertSerializer.Meta):
        read_only_fields = exclude(PcCertSerializer.Meta.read_only_fields, 'common_name')
        required = exclude(PcCertSerializer.Meta.required, 'request')

    request = serializers.CharField(
        required=False,
        trim_whitespace=False,
        style={'base_template': 'textarea.html'},
    )


class CourtecyVPNCertSerializer(LinuxPcCertSerializer):
    class Meta(LinuxPcCertSerializer.Meta):
        pass


class LinuxTokenCertSerializer(PcCertSerializer, SelfRequestOnlyMixin):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        fields = BaseMeta.fields + ('request', 'helpdesk_ticket', )
        required = ('request',)


class AssessorCertSerializer(AssessorCertMixin, PcCertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')


class MobvpnCertSerializer(MobvpnCertMixin, PcCertSerializer):
    class Meta(BaseMeta):
        fields = BaseMeta.fields + (
            'request', 'download_ca_pem', 'download_tls_key', 'download_config_ovpn', 'helpdesk_ticket',
        )
        required = ('request',)

    download_ca_pem = relations.HyperlinkedIdentityField(
        view_name='api:certificate-ca-pem',
        lookup_field='pk',
        read_only=True,
    )
    download_tls_key = relations.HyperlinkedIdentityField(
        view_name='api:certificate-tls-key',
        lookup_field='pk',
        read_only=True,
    )
    download_config_ovpn = relations.HyperlinkedIdentityField(
        view_name='api:certificate-config-ovpn',
        lookup_field='pk',
        read_only=True,
    )


class RcServerCertSerializer(RcServerCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        required = ('common_name',)


class YcServerCertSerializer(YcServerCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        fields = BaseMeta.fields + ('hosts',)

    hosts = HostSerializer(required=False, many=True, style={'base_template': 'input.html'})


class MdbCertSerializer(MdbCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        fields = BaseMeta.fields + ('hosts',)

    hosts = HostSerializer(required=False, many=True, style={'base_template': 'input.html'})


class VpnTokenCertSerializer(VpnTokenCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        fields = BaseMeta.fields + ('request', 'helpdesk_ticket',)
        required = ('request',)


class Vpn1DCertSerializer(Vpn1DCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        fields = BaseMeta.fields + ('helpdesk_ticket',)
        required = ('common_name',)


class ZombieCertSerializer(ZombieCertMixin, PcCertSerializer):
    class Meta(BaseMeta):
        read_only_fields = exclude(BaseMeta.read_only_fields, 'common_name')
        fields = BaseMeta.fields + (
            'request', 'helpdesk_ticket', 'pc_inum', 'pc_serial_number', 'pc_hostname', 'pc_mac',
        )


class TpmSmartcard1CCertSerializer(TpmSmartcard1CCertMixin, BaseV2CertSerializer):
    class Meta(BaseMeta):
        fields = BaseMeta.fields + ('request', 'helpdesk_ticket',)
        required = ('request',)
