from django.http import HttpResponseBadRequest
from django.utils.datastructures import MultiValueDict
from rest_framework import generics, exceptions, renderers

from intranet.crt.api.base.options import DEFAULT_API_OPTIONS
from intranet.crt.api.v1.certificates.filters import CertificateFilter
from intranet.crt.api.v1.certificates.views import certificates
from intranet.crt.api.v2.certificates.permissions import gather_permissions
from intranet.crt.api.v2.certificates.renderers import FormlessBrowsableAPIRenderer
from intranet.crt.api.v1.certificates.renderers import (
    PEMRenderer,
    PFXRenderer,
    IphoneRenderer,
)
from intranet.crt.api.v2.certificates.serializers.dispatch import CertificateSerializerDispatcher, get_serializer

from intranet.crt.constants import ACTION_TYPE
from intranet.crt.core.models import Certificate


V2_API_OPTIONS = DEFAULT_API_OPTIONS


class WithApiOptionsMixin(object):
    def get_serializer(self, *args, **kwargs):
        kwargs.setdefault('api_options', self.API_OPTIONS)

        serializer_class = self.get_serializer_class()
        kwargs['context'] = self.get_serializer_context()
        return serializer_class(*args, **kwargs)


class CertificateList(WithApiOptionsMixin, certificates.CertificateList):
    API_OPTIONS = V2_API_OPTIONS
    serializer_class = CertificateSerializerDispatcher
    renderer_classes = [
        renderers.JSONRenderer,
        FormlessBrowsableAPIRenderer,
    ]

    def get_list_serializer(self, data):
        return self.serializer_class(
            data,
            many=True,
            context=self.get_serializer_context(),
            api_options=self.API_OPTIONS,
        )

    def get_serializer_class(self):
        cert_type_name = self.request.data.get('type')
        ca_name = self.request.data.get('ca_name')
        if cert_type_name and ca_name:
            return get_serializer(cert_type_name, ca_name)
        else:
            return self.serializer_class

    def _parse_params(self):
        self.params = MultiValueDict(self.request.query_params.lists())
        self.params.pop('username', None)  # Используется только в v1

    def get_queryset(self):
        if 'order_by' in self.params:
            ordering = self.params['order_by']
        else:
            ordering = '-added'

        if 'serial_number' in self.params:
            self.params['serial_number'] = self.params['serial_number'].upper()

        query = (
            Certificate.objects
            .for_user(self.request.user)
            .order_by(ordering)
            .select_related(
                'user',
                'requester',
                'type',
                'abc_service',
            )
            .prefetch_related(
                'hosts',
                'hosts_to_approve',
                'approve_request',
            )
            .prefetch_tags(only_active=False)
        )
        return CertificateFilter(self.params, query).qs


class RetrieveWithOptionsView(WithApiOptionsMixin, generics.RetrieveAPIView):
    pass


class CertificateDetail(RetrieveWithOptionsView):
    API_OPTIONS = V2_API_OPTIONS
    model = Certificate
    queryset = Certificate.objects.all()
    serializer_class = CertificateSerializerDispatcher
    # TODO(v-sopov): сейчас здесь наблюдается неконсистентность относительно /certificates
    # TODO(v-sopov): для хостовых сертификатов (хождение в голем vs cert.user)
    action_names = ['update']
    permission_classes = gather_permissions(action_names)

    def post(self, request, *args, **kwargs):
        raise exceptions.MethodNotAllowed('POST')

    def patch(self, request, *args, **kwargs):
        cert = self.get_object()
        certificates.update_certificate(request, cert)
        cert.refresh_from_db()
        return self.retrieve(request, *args, **kwargs)


class CertificateDownload(WithApiOptionsMixin, certificates.CertificateDownload):
    API_OPTIONS = V2_API_OPTIONS
    action_names = ['download']
    permission_classes = gather_permissions(action_names)
    renderer_classes = [PEMRenderer, PFXRenderer, IphoneRenderer]

    def get(self, request, *args, **kwargs):
        if not request.query_params.get('format'):
            message = '\'format\' parameter should be specified'
            return HttpResponseBadRequest(message.encode('utf-8'))
        return super(CertificateDownload, self).get(request, *args, **kwargs)


class CertificateOperations(RetrieveWithOptionsView):
    API_OPTIONS = V2_API_OPTIONS
    model = Certificate
    queryset = Certificate.objects.all()
    serializer_class = CertificateSerializerDispatcher
    action_names = ['hold', 'unhold', 'revoke']
    permission_classes = gather_permissions(action_names)

    def do_revoke(self, request, cert):
        cert.controller.revoke(request.user, description='by api(v2)')

    def do_hold(self, request, cert):
        cert.controller.hold(request.user, description='by api(v2)')

    def do_unhold(self, request, cert):
        cert.controller.unhold(request.user, description='by api(v2)')

    def post(self, request, *args, **kwargs):
        action = request.data.get('action')
        handler = getattr(self, 'do_{}'.format(action), None)
        if handler is None:
            raise exceptions.ParseError('Invalid action "{}"'.format(action))

        cert = self.get_object()
        handler(request, cert)

        return self.retrieve(request, *args, **kwargs)
