from rest_framework import serializers

from intranet.crt.api.v1.certificates import fields
from intranet.crt.constants import CERT_STATUS, CERT_TYPE


class ExtendableFieldMixin(object):
    pass


class SuggestableField(serializers.Field):
    def __init__(self, *args, **kwargs):
        assert hasattr(self, 'suggest_path'), 'SuggestableField subclass should have suggest_path'
        super(SuggestableField, self).__init__(*args, **kwargs)


class CaNameField(SuggestableField, serializers.CharField):
    suggest_path = 'frontend:suggest:ca_names'


class UserField(SuggestableField, serializers.SlugRelatedField):
    suggest_path = 'frontend:suggest:users'

    def to_representation(self, instance):
        return {
            'username': instance.username,
            'first_name': {
                'ru': instance.first_name_ru,
                'en': instance.first_name,
            },
            'last_name': {
                'ru': instance.last_name_ru,
                'en': instance.last_name,
            },
            'is_active': instance.is_active,
            'in_hiring': instance.in_hiring,
        }


class CertTypeField(SuggestableField, serializers.SlugRelatedField, ExtendableFieldMixin):
    suggest_path = 'frontend:suggest:cert_types'

    def to_representation(self, instance):
        return {
            'value': instance.name,
            'human': {
                'ru': CERT_TYPE.humanized[instance.name]['ru'],
                'en': CERT_TYPE.humanized[instance.name]['en'],
            },
        }


class CertStatusField(SuggestableField, serializers.CharField, ExtendableFieldMixin):
    suggest_path = 'frontend:suggest:statuses'

    def to_representation(self, value):
        return {
            'value': value,
            'human': {
                'ru': CERT_STATUS.humanized[value]['ru'],
                'en': CERT_STATUS.humanized[value]['en'],
            }
        }


class AbcServiceField(SuggestableField, fields.AbcServiceField):
    suggest_path = 'frontend:suggest:abc_services'

    def to_representation(self, instance):
        return {
            'id': instance.external_id,
            'slug': instance.slug,
            'name': {
                'ru': instance.name,
                'en': instance.name_en,
            },
        }
