from collections import OrderedDict
from cryptography.hazmat._oid import ObjectIdentifier

from django.conf import settings
from model_utils import Choices

DAYS_IN_WEEK = settings.DAYS_IN_WEEK
DAYS_IN_MONTH = settings.DAYS_IN_MONTH
DAYS_IN_YEAR = settings.DAYS_IN_YEAR
SECONDS_IN_MINUTE = settings.SECONDS_IN_MINUTE
SECONDS_IN_HOUR = settings.SECONDS_IN_HOUR

DAYS_FOR_CERTIFICATE_METRICS = settings.DAYS_FOR_CERTIFICATE_METRICS

STAFF_SYNC_LIMIT = 1000
REMOVE_USERS_FROM_SCOPE_THRESHOLD = 0.5  # при удалении более 50% пользователей из скоупа прекращаем синхронизацию
SKIP_INCONSISTENT_GROUPS_THRESHOLD = 0.01  # при пропуске более 1% групп прекращаем синхронизацию
USE_THRESHOLD_IF_USERS_GREATER = 3
ABC_CERTIFICATE_MANAGER_SCOPE = 'cert'
ABC_ADMINISTRATOR_SCOPE = 'administration'


class CERT_TYPE(object):
    HOST = 'host'
    PC = 'pc'
    BANK_PC = 'bank-pc'
    ZOMB_PC = 'pc-client'
    COURTECY_VPN = 'courtecy-vpn'
    LINUX_PC = 'linux-pc'
    LINUX_TOKEN = 'linux-token'
    MOBVPN = 'mobvpn'
    BOTIK = 'botik'
    NINJA = 'ninja'
    NINJA_EXCHANGE = 'ninja-exchange'
    HYPERCUBE = 'hypercube'
    ASSESSOR = 'assessor'
    RC_SERVER = 'rc-server'
    YC_SERVER = 'yc-server'
    CLIENT_SERVER = 'client-server'
    BANK_CLIENT_SERVER = 'bank-client-server'
    VPN_TOKEN = 'vpn-token'
    IMDM = 'imdm'
    WIN_PC_AUTO = 'win-pc-auto'
    WIN_WH_SHARED = 'win-wh-shared'
    ZOMBIE = 'zombie'
    POSTAMATE = 'postamate'
    SDC = 'sdc'
    MDB = 'mdb'
    TPM_SMARTCARD_1C = 'tpm-smartcard-1c'
    TEMP_PC = 'temp-pc'
    VPN_1D = 'vpn-1d'

    # Inactive types:
    SMIME = 'smime'
    CODE_SIGN = 'code-sign'

    ALL_TYPES = frozenset({
        HOST, PC, BANK_PC, ZOMB_PC, COURTECY_VPN, LINUX_PC, LINUX_TOKEN, MOBVPN, BOTIK, NINJA,
        NINJA_EXCHANGE, HYPERCUBE, ASSESSOR, RC_SERVER, YC_SERVER, CLIENT_SERVER, BANK_CLIENT_SERVER, VPN_TOKEN,
        IMDM, WIN_PC_AUTO, ZOMBIE, SDC, POSTAMATE, MDB, SMIME, CODE_SIGN, TPM_SMARTCARD_1C, TEMP_PC,
        WIN_WH_SHARED, VPN_1D,
    })

    INACTIVE_TYPES = frozenset({SMIME, CODE_SIGN})

    # Certificate types that require 'core.can_issue_device_certificates' permission
    DEVICE_TYPES = frozenset({PC, LINUX_TOKEN, COURTECY_VPN, ASSESSOR, ZOMBIE, VPN_TOKEN})

    # Certificate types that are installed on non-user hardware, i.e. servers and network routers
    HARDWARE_TYPES = frozenset({HOST, CLIENT_SERVER, BANK_CLIENT_SERVER, RC_SERVER, YC_SERVER, SDC, POSTAMATE, MDB})

    # Certificate types for users hardware
    USER_HARDWARE_TYPES = frozenset({PC, LINUX_PC, BANK_PC})

    # Certificate types that user without any special permission or role can request
    PUBLIC_REQUEST_TYPES = frozenset({LINUX_PC, MOBVPN, BOTIK, NINJA, NINJA_EXCHANGE, HOST})

    # Certificate types that can be requested with CSR
    CSR_REQUESTABLE_TYPES = frozenset({
        LINUX_TOKEN, CLIENT_SERVER, BANK_CLIENT_SERVER, VPN_TOKEN, MOBVPN, PC, BANK_PC, IMDM,
        ZOMBIE, LINUX_PC, ZOMB_PC, COURTECY_VPN, TPM_SMARTCARD_1C, TEMP_PC,
        HOST  # temporarily
    })

    # Certificate types that can be requested without CSR given common_name
    NOCSR_REQUESTABLE_TYPES = frozenset({
        COURTECY_VPN, LINUX_PC, BOTIK, NINJA, NINJA_EXCHANGE, HYPERCUBE,
        RC_SERVER, YC_SERVER, ASSESSOR, SDC, POSTAMATE, MDB,
    })

    # Certificates that can not be requested from API
    NO_API_TYPES = {WIN_PC_AUTO, WIN_WH_SHARED}

    # Certificates that can be revoke with permissions `can_revoke_users_certificates`
    USERS_TYPES = frozenset({
        PC, BANK_PC, ZOMB_PC, LINUX_PC, LINUX_TOKEN, MOBVPN, BOTIK, NINJA, COURTECY_VPN,
        NINJA_EXCHANGE, VPN_TOKEN, IMDM, WIN_PC_AUTO, ZOMBIE, TPM_SMARTCARD_1C, TEMP_PC,
        WIN_WH_SHARED, VPN_1D,
    })

    # Certificates without certificate-user relation
    NO_USER_TYPES = frozenset({WIN_PC_AUTO, WIN_WH_SHARED})

    # Certificates types to authenticate in Yandex Intranet
    INTRANET_ACCESS_TYPES = frozenset({
        PC, BANK_PC, LINUX_PC, LINUX_TOKEN, MOBVPN, NINJA, NINJA_EXCHANGE, VPN_TOKEN, IMDM, VPN_1D,
        WIN_PC_AUTO, WIN_WH_SHARED, ZOMBIE, POSTAMATE, HYPERCUBE, COURTECY_VPN, TEMP_PC,
    })
    TAGGABLE_TYPES = INTRANET_ACCESS_TYPES

    CUSTOM_COMMON_NAME_TYPES = {TPM_SMARTCARD_1C}

    USERNAME_IN_COMMON_NAME_TYPES = frozenset({
        COURTECY_VPN, ZOMBIE, NINJA, ASSESSOR, VPN_TOKEN, TPM_SMARTCARD_1C, TEMP_PC,
        LINUX_PC, LINUX_TOKEN, PC, BANK_PC, MOBVPN, HYPERCUBE, NINJA_EXCHANGE, IMDM, VPN_1D,
    })

    REISSUABLE_TYPES = frozenset({
        BANK_PC, CLIENT_SERVER, BANK_CLIENT_SERVER, LINUX_PC, PC, RC_SERVER, TPM_SMARTCARD_1C, VPN_TOKEN, ZOMB_PC,
    })
    CLIENT_AUTH_ALLOWED_TYPES = REISSUABLE_TYPES

    humanized = OrderedDict((
        (HOST, {'en': 'For web servers', 'ru': 'Для web-сервера'}),
        (PC, {'en': 'For working computers', 'ru': 'Для рабочего компьютера'}),
        (BANK_PC, {'en': 'For bank computers', 'ru': 'Для компьютера банка'}),
        (ZOMB_PC, {'en': 'For machines and zombies', 'ru': 'Для машин и зомбиков'}),
        (COURTECY_VPN, {'en': '30days vpn', 'ru': '30 дней VPN'}),
        (LINUX_PC, {'en': 'For Linux computers', 'ru': 'Для машин под Linux'}),
        (LINUX_TOKEN, {'en': 'For tokens', 'ru': 'Для токенов'}),
        (MOBVPN, {'en': 'For mobile VPN', 'ru': 'Мобильный VPN'}),
        (BOTIK, {'en': 'For botik', 'ru': 'Для ботика'}),
        (NINJA, {'en': 'For PDAS', 'ru': 'Для сети PDAS'}),
        (NINJA_EXCHANGE, {'en': 'For PDAS with exchange', 'ru': 'Для сети PDAS с эксченьжем'}),
        (HYPERCUBE, {'en': 'For Hypercube devices', 'ru': 'Для устройств из Hypercube'}),
        (ASSESSOR, {'en': 'For testing assessors', 'ru': 'Для тестирующих асессоров'}),
        (RC_SERVER, {'en': 'For Runtime Cloud', 'ru': 'Для серверов Runtime Cloud'}),
        (YC_SERVER, {'en': 'For Yandex.Cloud', 'ru': 'Для серверов Yandex.Cloud'}),
        (CLIENT_SERVER, {'en': 'For NOC machines', 'ru': 'Для машин NOCов'}),
        (BANK_CLIENT_SERVER, {'en': 'For bank machines', 'ru': 'Для машин инфраструктуры банков'}),
        (VPN_TOKEN, {'en': 'For VPN-tokens', 'ru': 'Для VPN-токенов'}),
        (IMDM, {'en': 'For mobile Apple devices', 'ru': 'Для мобильных устройств Apple'}),
        (WIN_PC_AUTO, {
            'en': 'Autorequestable for Win-machines',
            'ru': 'Автозапрашиваемый для Win-машин',
        }),
        (WIN_WH_SHARED, {
            'en': 'Autorequestable non-user for Win machines in warehouses',
            'ru': 'Автозапрашиваемый для неименных win машин на складах',
        }),
        (ZOMBIE, {'en': 'For zombie authorization', 'ru': 'Для авторизации зомбиков'}),
        (SDC, {'en': 'For self-driving cars', 'ru': 'Для беспилотных автомобилей'}),
        (POSTAMATE, {'en': 'For postamates', 'ru': 'Для постаматов'}),
        (MDB, {'en': 'For MDB', 'ru': 'Для серверов MDB'}),
        (TPM_SMARTCARD_1C, {'en': 'For TPM Smartcard (1C)', 'ru': 'Для TPM смарткард (1C)'}),
        (TEMP_PC, {'en': 'For temporary notebooks', 'ru': 'Для временных ноутбуков'}),
        (VPN_1D, {'en': 'For one day VPN access', 'ru': 'Для доступа к VPN на один день'}),
        (SMIME, {'en': 'SMIME (inactive)', 'ru': 'SMIME (неактивен)'}),
        (CODE_SIGN, {'en': 'Code-Sign (inactive)', 'ru': 'Code-Sign (неактивен)'}),
    ))

    HUMAN_PREFIX = {
        HOST: {'en': 'host', 'ru': 'хост'},
        CLIENT_SERVER: {'en': 'host', 'ru': 'клиент-сервер'},
        BANK_CLIENT_SERVER: {'en': 'bank-host', 'ru': 'клиент-сервер банка'},
    }

    @classmethod
    def choices(cls):
        return tuple([
            (key, value['ru'])
            for key, value in cls.humanized.items()
            if key not in cls.INACTIVE_TYPES
        ])

    @classmethod
    def active_types(cls):
        return [key for key in cls.humanized.keys() if key not in cls.INACTIVE_TYPES]


class CERT_STATUS(object):
    REQUESTED = 'requested'
    VALIDATION = 'validation'
    ISSUED = 'issued'
    REVOKED = 'revoked'
    HOLD = 'hold'
    ERROR = 'error'
    EXPIRED = 'expired'
    REVOKING = 'revoking'
    NEED_APPROVE = 'need_approve'
    REJECTED = 'rejected'

    humanized = OrderedDict((
        (REQUESTED, {'en': 'Requested', 'ru': 'Запрошен'}),
        (VALIDATION, {'en': 'Waiting for domain validation', 'ru': 'Ожидает подтверждения домена'}),
        (ISSUED, {'en': 'Issued', 'ru': 'Выдан'}),
        (REVOKED, {'en': 'Revoked', 'ru': 'Отозван'}),
        (HOLD, {'en': 'On hold', 'ru': 'Заморожен'}),
        (ERROR, {'en': 'Error', 'ru': 'Произошла ошибка'}),
        (EXPIRED, {'en': 'Expired', 'ru': 'Истёк'}),
        # актуально для внутренних сертификатов, для которых нет прямой возможности отозвать
        (REVOKING, {'en': 'Revoking', 'ru': 'В процессе отзыва'}),
        (NEED_APPROVE, {'en': 'Needs infosec approve', 'ru': 'Требует подтверждения СИБ'}),
        (REJECTED, {'en': 'Rejected', 'ru': 'Отклонён'}),
    ))

    @classmethod
    def choices(cls):
        return tuple([(key, value['ru']) for key, value in cls.humanized.items()])

    @classmethod
    def all_statuses(cls):
        return list(cls.humanized.keys())

    RENDERABLE_STATUSES = frozenset({ISSUED, REVOKED, HOLD, EXPIRED, REVOKING})
    DEAD_STATUSES = frozenset({REVOKED, ERROR, EXPIRED, REVOKING, REJECTED})  # уже никогда не будут активными
    ALIVE_STATUSES = frozenset(set(humanized.keys()) - DEAD_STATUSES)  # или активные, или могут такими стать
    TAGGABLE_STATUSES = frozenset({REQUESTED, ISSUED})


class CERT_ACTION(object):
    UPDATE = 'update'
    DOWNLOAD = 'download'
    HOLD = 'hold'
    UNHOLD = 'unhold'
    REVOKE = 'revoke'

    humanized = {
        UPDATE: {'ru': 'Обновить', 'en': 'Update'},
        DOWNLOAD: {'ru': 'Скачать', 'en': 'Download'},
        HOLD: {'ru': 'Заморозить', 'en': 'Hold'},
        UNHOLD: {'ru': 'Разморозить', 'en': 'Unhold'},
        REVOKE: {'ru': 'Отозвать', 'en': 'Revoke'},
    }

    available_by_status = {
        CERT_STATUS.REQUESTED: [UPDATE],
        CERT_STATUS.VALIDATION: [UPDATE],
        CERT_STATUS.ISSUED: [UPDATE, DOWNLOAD, HOLD, REVOKE],
        CERT_STATUS.REVOKED: [],
        CERT_STATUS.HOLD: [UPDATE, DOWNLOAD, UNHOLD, REVOKE],
        CERT_STATUS.ERROR: [],
        CERT_STATUS.EXPIRED: [],
        CERT_STATUS.REVOKING: [UPDATE, DOWNLOAD],
        CERT_STATUS.NEED_APPROVE: [UPDATE],
        CERT_STATUS.REJECTED: [],

    }

    @classmethod
    def humanize(cls, action):
        return {
            'id': action,
            'name': {
                'ru': cls.humanized[action]['ru'],
                'en': cls.humanized[action]['en'],
            }
        }

    @classmethod
    def all_actions(cls):
        return [cls.UPDATE, cls.DOWNLOAD, cls.HOLD, cls.UNHOLD, cls.REVOKE]


class ACTION_TYPE(object):
    CERT_REVOKE = 'cert_revoke'
    CERT_HOLD = 'cert_hold'
    CERT_UNHOLD = 'cert_unhold'
    CERT_ADD_TAG = 'tag_add_cert'
    CERT_REMOVE_TAG = 'tag_remove_cert'
    CERT_CHANGE_ABC_SERVICE = 'cert_change_abc_service'
    CERT_ADD_TO_HOLD_QUEUE = 'cert_add_to_hold_queue'
    CERT_MARK_EXPIRED = 'cert_mark_expired'
    CERT_REQUEST_APPROVED = 'cert_request_approved'
    CERT_REQUEST_REJECTED = 'cert_request_rejected'
    CERT_PRIVATE_KEY_DELETED = 'cert_private_key_deleted'

    TAG_FILTER_ADD_USER = 'tag_filter_add_user'
    TAG_FILTER_REMOVE_USER = 'tag_filter_remove_user'
    TAG_FILTER_MARKED_BROKEN = 'tag_filter_marked_broken'
    TAG_FILTER_MARKED_NOT_BROKEN = 'tag_filter_marked_not_broken'
    TAG_FILTER_CHANGED = 'tag_filter_changed'

    @classmethod
    def choices(cls):
        return (
            (cls.CERT_REVOKE, 'Certificate revoke'),
            (cls.CERT_HOLD, 'Certificate hold'),
            (cls.CERT_UNHOLD, 'Certificate unhold'),
            (cls.CERT_ADD_TAG, 'Certificate add tag'),
            (cls.CERT_REMOVE_TAG, 'Certificate remove tag'),
            (cls.CERT_CHANGE_ABC_SERVICE, 'Certificate change ABC service'),
            (cls.CERT_ADD_TO_HOLD_QUEUE, 'Certificate added to hold queue'),
            (cls.CERT_MARK_EXPIRED, 'Certificate mark expired'),
            (cls.CERT_REQUEST_APPROVED, 'Certificate request approved'),
            (cls.CERT_REQUEST_REJECTED, 'Certificate request rejected'),
            (cls.CERT_PRIVATE_KEY_DELETED, 'Certificate private key deleted'),

            (cls.TAG_FILTER_ADD_USER, 'TagFilter add user'),
            (cls.TAG_FILTER_REMOVE_USER, 'TagFilter remove user'),
        )


class TASK_TYPE(object):
    SYNC_ABC_SERVICES = 'sync_abc_services'
    SYNC_GROUPS = 'sync_groups'
    SYNC_ASSESSOR_CERTS = 'sync_assessor_certs'
    SYNC_TAGS = 'sync_tags'
    SYNC_FILTER_TAGS = 'sync_filter_tags'
    SYNC_CERT_TYPE_TAGS = 'sync_cert_type_tags'
    SYNC_CLEAN_TAGS = 'sync clean tags'
    SYNC_CVS_TAGS = 'sync_cvs_tags'
    SYNC_USERS = 'sync_users'
    SYNC_CRL = 'sync_crl'
    MARK_EXPIRED = 'mark_expired'
    IMPORT_INTERNAL_CA_CERTS = 'import_internal_ca_certs'
    REVOKE_QUEUED_CERTS = 'revoke_queued_certs'
    REVOKE_DISMISSED = 'revoke_dismissed'
    REVOKE_DUPLICATE = 'revoke_duplicate'
    REMOVE_PRIVATE_KEYS = 'remove_private_keys'
    SAVE_KEYS_TO_YAV = 'save_private_keys_to_yav'
    DELETE_UPLOADED_PRIVATE_KEYS = 'delete_uploaded_private_keys'

    @classmethod
    def all(cls):
        return (
            cls.SYNC_ABC_SERVICES,
            cls.SYNC_GROUPS,
            cls.SYNC_ASSESSOR_CERTS,
            cls.SYNC_TAGS,
            cls.SYNC_FILTER_TAGS,
            cls.SYNC_CERT_TYPE_TAGS,
            cls.SYNC_CLEAN_TAGS,
            cls.SYNC_CVS_TAGS,
            cls.SYNC_USERS,
            cls.SYNC_CRL,
            cls.MARK_EXPIRED,
            cls.IMPORT_INTERNAL_CA_CERTS,
            cls.REVOKE_QUEUED_CERTS,
            cls.REVOKE_DISMISSED,
            cls.REVOKE_DUPLICATE,
            cls.REMOVE_PRIVATE_KEYS,
            cls.SAVE_KEYS_TO_YAV,
        )

    @classmethod
    def choices(cls):
        return (
            (cls.SYNC_ABC_SERVICES, 'Sync ABC services'),
            (cls.SYNC_GROUPS, 'Sync groups'),
            (cls.SYNC_ASSESSOR_CERTS, 'Sync assessor certs'),
            (cls.SYNC_TAGS, 'Sync tags'),
            (cls.SYNC_FILTER_TAGS, 'Sync filter tags'),
            (cls.SYNC_CERT_TYPE_TAGS, 'Sync cert type tags'),
            (cls.SYNC_CLEAN_TAGS, 'Sync clean tags'),
            (cls.SYNC_CVS_TAGS, 'Sync cvs tags'),
            (cls.SYNC_USERS, 'Sync users'),
            (cls.SYNC_CRL, 'Sync revokes / hold certs with crl'),
            (cls.MARK_EXPIRED, 'Mask expired certs'),
            (cls.IMPORT_INTERNAL_CA_CERTS, 'Import InternalCA certs'),
            (cls.REVOKE_QUEUED_CERTS, 'Revoke queued certs'),
            (cls.REVOKE_DISMISSED, 'Add dismissed users certificates into revoke queue'),
            (cls.REVOKE_DUPLICATE, 'Add duplicate certificates into revoke queue'),
            (cls.REMOVE_PRIVATE_KEYS, 'Removing old private keys'),
            (cls.SAVE_KEYS_TO_YAV, 'Save private keys to yav'),
        )


class TAG_FILTER_TYPE(object):
    STAFF_API = 'staff_api'
    STAFF_FILTER = 'staff_filter'
    IDM_SYSTEM = 'idm_system'

    SYNCABLE_TYPES = frozenset({STAFF_API, STAFF_FILTER})

    @classmethod
    def choices(cls):
        return [
            (cls.STAFF_API, 'API Staff'),
            (cls.STAFF_FILTER, 'Staff фильтр'),
            (cls.IDM_SYSTEM, 'IDM'),
        ]


class TAG_TYPE(object):
    NOC_USE = 'noc_use'

    @classmethod
    def choices(cls):
        return [
            (cls.NOC_USE, 'Используется NOC'),
        ]


class TAG_PREFIX(object):
    WIRED = 'Office.8021X.'
    WIRED_ADDITIONAL = 'wired.'
    WIFI = 'Office.WiFi.'
    VPN = 'Office.VPN.'

    WIRED_PREFIXES = (WIRED, WIRED_ADDITIONAL)


class TAG_NAME(object):
    VPN = 'Office.VPN'
    TMPAUTH = 'TmpAuth'
    DEFAULT_NAME = 'Default'


class CA_NAME(object):
    INTERNAL_CA = 'InternalCA'
    INTERNAL_TEST_CA = 'InternalTestCA'
    RC_INTERNAL_CA = 'RcInternalCA'
    YC_INTERNAL_CA = 'YcInternalCA'
    CB_INTERNAL_CA = 'CbInternalCA'
    CERTUM_TEST_CA = 'CertumTestCA'
    CERTUM_PRODUCTION_CA = 'CertumProductionCA'
    GLOBALSIGN_TEST_CA = 'GlobalSignTestCA'
    GLOBALSIGN_PRODUCTION_CA = 'GlobalSignProductionCA'
    TEST_CA = 'TestCA'
    APPROVABLE_TEST_CA = 'ApprovableTestCA'

    EXTERNAL_CAS = (CERTUM_PRODUCTION_CA, GLOBALSIGN_PRODUCTION_CA)
    EXTERNAL_TEST_CAS = (CERTUM_TEST_CA, GLOBALSIGN_TEST_CA)

    CERTUM_CAS = (CERTUM_TEST_CA, CERTUM_PRODUCTION_CA)
    GLOBALSIGN_CAS = (GLOBALSIGN_TEST_CA, GLOBALSIGN_PRODUCTION_CA)

    @classmethod
    def choices(cls):
        return [
            (cls.INTERNAL_CA, 'Внутренний CA'),
            (cls.INTERNAL_TEST_CA, 'Внутренний тестовый CA'),
            (cls.RC_INTERNAL_CA, 'Subordinate enterprise CA для нужд runtime cloud'),
            (cls.YC_INTERNAL_CA, 'Subordinate enterprise CA for Yandex.Cloud operations'),
            (cls.CB_INTERNAL_CA, 'Internal CA for Hypercube'),
            (cls.CERTUM_TEST_CA, 'Certum (testing, defunct)'),
            (cls.CERTUM_PRODUCTION_CA, 'Certum (production)'),
            (cls.GLOBALSIGN_TEST_CA, 'GlobalSign  (testing)'),
            (cls.GLOBALSIGN_PRODUCTION_CA, 'GlobalSign (production)'),
            (cls.TEST_CA, 'Self-signed'),
            (cls.APPROVABLE_TEST_CA, 'Self-signed, requires approval'),
        ]

    @classmethod
    def all_ca(cls):
        return sorted(ca for (ca, desc) in cls.choices())

    CUSTOM_EXTENSION_SUPPORT = frozenset({INTERNAL_CA, INTERNAL_TEST_CA, TEST_CA})


class CERT_TEMPLATE(object):
    WEB_SERVER = 'WebServer-auto'
    WEB_SERVER_3D = 'WebServer-auto-3days'
    WEB_SERVER_ECC = 'WebServer-auto-ecc'
    WEB_SERVER_ECC_3D = 'WebServer-auto-ecc-3days'
    USER_COMMON = 'yaUser_common'
    USER_BANK = 'yaUser_bank'
    USER_LINUX = 'yaUser_linux'
    VPN_1D = 'yaUser_linux_1day'
    USER_LINUX_EXT = 'yaUser_linux_ext'
    COURTECY_VPN = 'YaCourtecyCertForVPN'
    MOBVPN = 'yaUser_mobvpn'
    BOTANIK = 'yaBotanik'
    USER_PDAS = 'YaUser-PDAS'
    USER_PDAS_1D = 'YaUser-PDAS1'
    USER_PDAS_3D = 'YaUser-PDAS3'
    USER_PDAS_5D = 'YaUser-PDAS5'
    USER_PDAS_10D = 'YaUser-PDAS10'
    USER_PDAS_30D = 'YaUser-PDAS30'
    ASSESSOR = 'yaAssessor'
    RC_SERVER = 'yaRC-Server'
    RC_SERVER_2W = 'yaRC-Server_2w'
    RC_SERVER_1M = 'yaRC-Server_1m'
    RC_SERVER_3M = 'yaRC-Server_3m'
    RC_SERVER_6M = 'yaRC-Server_6m'
    RC_SERVER_1Y = 'yaRC-Server_1y'
    YC_SERVER = 'yaCL-Server'
    CLIENT_SERVER = 'client-servercomputerAuto'
    BANK_CLIENT_SERVER = 'client-servercomputerAuto'  # CERTOR-2126
    SMART_CARD = 'YaSmartCardForVPN'
    USER_IMDM = 'YaUser-iMDM'
    WIN_PC_AUTO = 'Компьютер-auto-New'
    WIN_WH_SHARED = 'Win-wh-shared'
    ZOMBIE = 'yaZombie'
    SDC = 'yaSelfDrivingCar'
    POSTAMATE = 'yaPostamate'
    MDB = 'yaMdb'
    TPM_SMARTCARD_1C = 'tpm_smartcard_logon(for1c)'


class TAG_SOURCE(object):
    MANUAL = 'manual'
    FILTERS = 'filters'
    CERT_TYPE = 'cert_type'
    IDM = 'idm'

    @classmethod
    def choices(cls):
        return [
            (cls.MANUAL, 'By api/db'),
            (cls.FILTERS, 'By filters'),
            (cls.CERT_TYPE, 'By cert type'),
            (cls.IDM, 'By IDM \'tag\' role'),
        ]


class CERT_DETAIL_ACTION(object):
    HOLD = 'hold'
    UNHOLD = 'unhold'
    REVOKE = 'revoke'
    UPDATE = 'update'


class CRT_GROUP_TYPE(object):
    DEPARTMENT = 'department'
    SERVICE = 'service'
    SERVICE_ROLE = 'servicerole'
    WIKI = 'wiki'

    @classmethod
    def choices(cls):
        return [
            (cls.DEPARTMENT, 'Department'),
            (cls.SERVICE, 'Service group'),
            (cls.SERVICE_ROLE, 'Service role scope'),
            (cls.WIKI, 'Wiki-group'),
        ]


class CERT_EXTENSION:
    WIRED_TAGS = 'wired_tags'
    WIRELESS_TAGS = 'wireless_tags'
    VPN_TAGS = 'vpn_tags'
    MOBILE_TAGS = 'mobile_tags'

    OID_STRING = {
        WIRED_TAGS: '1.3.6.1.4.1.11069.13238.8021.0',
        WIRELESS_TAGS: '1.3.6.1.4.1.11069.13238.8021.1',
        VPN_TAGS: '1.3.6.1.4.1.11069.13238.8021.2',
        MOBILE_TAGS: '1.3.6.1.4.1.11069.13238.8021.3',
    }
    OID = {
        WIRED_TAGS: ObjectIdentifier(OID_STRING[WIRED_TAGS]),
        WIRELESS_TAGS: ObjectIdentifier(OID_STRING[WIRELESS_TAGS]),
        VPN_TAGS: ObjectIdentifier(OID_STRING[VPN_TAGS]),
        MOBILE_TAGS: ObjectIdentifier(OID_STRING[MOBILE_TAGS]),
    }

    ALL = [WIRED_TAGS, WIRELESS_TAGS, VPN_TAGS, MOBILE_TAGS]

    PERMITTED_IN_PUBLIC_CSR = {
        CERT_TYPE.TEMP_PC: {
            WIRED_TAGS: {'Other'},
            WIRELESS_TAGS: {'PDAS'},
        },
    }


HOST_VALIDATION_CODE_STATUS = Choices(
    'validation',
    'validated',
    'error',
)


class AFFILIATION():
    YANDEX = 'yandex'
    YAMONEY = 'yamoney'
    EXTERNAL = 'external'
    OTHER = 'other'

    CHOICES = (
        (YANDEX, YANDEX),
        (YAMONEY, YAMONEY),
        (EXTERNAL, EXTERNAL),
        (OTHER, OTHER),
    )


class INCONSISTENCY_TYPE:
    GROUP = 'group'
    USER = 'user'

    @classmethod
    def choices(cls):
        return [
            (cls.GROUP, 'Group'),
            (cls.USER, 'User'),
        ]
