from django.conf import settings
from django.contrib import admin
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from intranet.crt.api.v1.approverequests.serializers import ApproveRequestSerializer
from intranet.crt.constants import CERT_STATUS, TAG_SOURCE
from intranet.crt.core.models import Certificate, CertificateType, ApproveRequest, HostToApprove
from intranet.crt.tags.models import CertificateTagRelation
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.admin import NoDeleteModelAdmin


class CertificateTypeAdmin(admin.ModelAdmin):
    list_display = ('name', 'is_active', 'description')

    def get_queryset(self, request):
        return CertificateType.objects.all()


class ApproveRequestAdmin(NoDeleteModelAdmin):
    list_display = ['common_name', 'status', 'approved', 'create_date',
                    'extended_validation', 'is_ecc', 'requester', 'issue_link', 'abc_service']
    list_select_related = ['certificate', 'certificate__requester', 'certificate__abc_service']
    actions = ['approve', 'reject']
    list_filter = (
        ('certificate__requester', admin.RelatedOnlyFieldListFilter),
    )

    @staticmethod
    def status(obj):
        return obj.certificate.status

    @staticmethod
    def common_name(obj):
        return mark_safe(
            '<a href="/console/#!/certificate/{cert.id}">{cert.common_name}</a>'
            .format(cert=obj.certificate)
        )

    @staticmethod
    def issue_link(obj):
        if obj.st_issue_key is None:
            return ''

        link = settings.CRT_STARTREK_ISSUE_URL.format(st_issue_key=obj.st_issue_key)
        return mark_safe(
            '<a href="{link}">{request.st_issue_key}</a>'.format(link=link, request=obj)
        )

    def has_add_permission(self, request):
        return False

    def get_queryset(self, request):
        return (
            ApproveRequest.objects
            .filter(certificate__status=CERT_STATUS.NEED_APPROVE)
            .select_related('certificate', 'certificate__requester')
        )

    def extended_validation(self, instance):
        return instance.certificate.extended_validation

    def is_ecc(self, instance):
        return instance.certificate.is_ecc

    def requester(self, instance):
        return instance.certificate.requester.username

    def approve(self, request, queryset):
        for item in queryset:
            serializer = ApproveRequestSerializer(
                item,
                context={'request': request},
                data={'approved': True}
            )
            serializer.is_valid()
            serializer.save()

        self.message_user(request, 'Requests approved')

    def reject(self, request, queryset):
        for item in queryset:
            serializer = ApproveRequestSerializer(
                item,
                context={'request': request},
                data={'approved': False},
            )
            serializer.is_valid()
            serializer.save()

        self.message_user(request, 'Requests rejected')

    def abc_service(self, instance):
        return instance.certificate.abc_service.name if instance.certificate.abc_service else '—'


class CertificateTagsInline(admin.TabularInline):
    model = CertificateTagRelation
    extra = 0

    def get_formset(self, *args, **kwargs):
        formset = super(CertificateTagsInline, self).get_formset(*args, **kwargs)
        formset.form.base_fields['source'].initial = TAG_SOURCE.MANUAL
        return formset


class CertificateAdmin(NoDeleteModelAdmin):
    list_display = ('id', 'common_name', 'serial_number', 'ca_name', 'status', 'abc_service')
    list_filter = ('ca_name', 'status')
    search_fields = ('common_name',)

    inlines = (CertificateTagsInline,)
    fields = list_display
    readonly_fields = ('id', 'common_name', 'serial_number', 'ca_name', 'status')


class CrtUserAdmin(NoDeleteModelAdmin):
    list_display = ('username', 'first_name', 'last_name', 'ninja_certs_quota')
    list_filter = ('is_staff', 'is_superuser', 'is_active', 'groups')
    search_fields = ('username', 'first_name', 'last_name')
    ordering = ('username',)
    filter_horizontal = ('groups', 'user_permissions',)

    fieldsets = (
        (None, {'fields': ('username', 'ninja_certs_quota')}),
        (_('Personal info'), {'fields': ('first_name', 'last_name',)}),
    )


admin.site.register(
    HostToApprove,
    exclude=['certificates'],
    list_display=('host', 'name_servers', 'globalsign_domain_id', 'managed_dns', 'auto_managed'),
    search_fields=('host',),
)
admin.site.register(CrtUser, CrtUserAdmin)
admin.site.register(Certificate, CertificateAdmin)
admin.site.register(CertificateType, CertificateTypeAdmin)
admin.site.register(ApproveRequest, ApproveRequestAdmin)
