import re
import logging

from django import http
from django.conf import settings
from django_yauth import middleware, exceptions
from django_yauth.user import YandexUser, YandexUserDescriptor, DjangoUserDescriptor, ApplicationDescriptor
from django_yauth.util import (get_current_host, get_passport_url,
                               debug_msg, get_setting, get_yauth_type)
from rest_framework.authentication import SessionAuthentication

from ylog.context import log_context

from intranet.crt.constants import CERT_STATUS, CERT_TYPE
from intranet.crt.utils.ldap import check_username_and_password_in_ldap
from intranet.crt.utils.random import generate_task_id
from intranet.crt.core.models import Certificate
from intranet.crt.users.models import CrtUser

log = logging.getLogger(__name__)


def is_technical(request):
    if request.path.startswith('/ping') or request.path.startswith('/monitorings'):
        return True


class CreateRequestLogger(object):
    def __init__(self, get_response):
        self.get_response = get_response

    def process_request(self, request):
        request.id = generate_task_id()
        if is_technical(request):
            return

        request.log_context = log_context(request_id=request.id)
        request.log_context.__enter__()
        with log_context(method=request.method, path=request.path):
            log.info('Request accepted')

    def process_response(self, request, response):
        if is_technical(request):
            return response

        content_type = response.get('content-type')
        with log_context(status_code=response.status_code, content_type=content_type):
            log.info('Request processed')
        request.log_context.__exit__(None, None, None)
        response['X-Request-Id'] = request.id
        return response

    def __call__(self, request):
        self.process_request(request)
        response = self.get_response(request)
        return self.process_response(request, response)


class SSLAuthMiddleware(object):
    def __init__(self, get_response):
        self.get_response = get_response

    def process_request(self, request):
        if request.environ.get('HTTP_X_SSL_CLIENT_VERIFY') != '0':
            return

        dn = request.environ.get('HTTP_X_SSL_CLIENT_SUBJECT')
        user = None
        login = settings.VALID_SSL_DN.get(dn, None)

        if login is None:
            serial = request.environ.get('HTTP_X_SSL_CLIENT_SERIAL')
            if serial is not None:
                qs = (
                    Certificate.objects.select_related('user').filter(
                        serial_number=serial,
                        status=CERT_STATUS.ISSUED,
                        ca_name__in=settings.CRT_CA_TO_AUTH_WITH,
                        type__name__in=CERT_TYPE.CLIENT_AUTH_ALLOWED_TYPES,
                    )
                )
                if qs.exists():
                    cert = qs.first()
                    login = cert.user.username
                    user = cert.user
        else:
            user = CrtUser.objects.filter(username=login).first()

        if user is None:
            request.environ['HTTP_X_SSL_CLIENT_VERIFY'] = 'NOT_KNOWN_TO_CRT'
            return

        request.yauser = YandexUser(1000000000, 0, {'login': login}, False, None, None)
        request.user = user

    def __call__(self, request):
        self.process_request(request)
        return self.get_response(request)


class CrossOrigin(object):
    """Эта миддльварь нужна для того, чтобы голем мог
    делать кросс-сайт запросы из разных сред, спродакшена и разработки.
    """

    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        response = self.get_response(request)

        origin = request.environ.get('HTTP_ORIGIN', '')

        for allowed in settings.ALLOWED_ORIGINS:
            if re.match(allowed, origin) is not None:
                response['Access-Control-Allow-Origin'] = origin
                response['Access-Control-Allow-Credentials'] = 'true'
                break

        return response


class LDAPAuthMiddleware(object):
    def __init__(self, get_response):
        self.get_response = get_response

    def process_request(self, request):
        ldap_username = request.META.get('HTTP_X_LDAP_USERNAME')
        ldap_password = request.META.get('HTTP_X_LDAP_PASSWORD')

        if not check_username_and_password_in_ldap(ldap_username, ldap_password):
            return

        # Нужно чтобы заставить test.Client работать с YandexUser, см /ninja
        request.yauser = YandexUser(1000000000, 0, {'login': ldap_username}, False, None, None)

    def __call__(self, request):
        self.process_request(request)
        return self.get_response(request)


class FuckedUpSessionAuthentication(SessionAuthentication):
    def enforce_csrf(self, request):
        """
        Отключаем здесь проверку csrf-токена, который случайно или нет был
        всегда до этого отключён.
        """


class LogCurrentUsername(object):
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        if not is_technical(request):
            try:
                login = request.yauser.login
            except exceptions.TwoCookiesRequired:
                login = '<NO_USER>'
            log.info('Current user: {}'.format(login))

        return self.get_response(request)


class NotAuthenticatedResponse(http.HttpResponse):
    status_code = 401


def get_retpath(request, add_params=None, del_params=None):
    """
    Скопипащенная из yauth версия get_current_url, которая удаляет /ninja/ из пути для ninja-доменов
    """
    host = get_current_host(request, drop_port=False)

    # Обрабатываем проксирование с ninja.y-t.ru/ на crt-api.y-t.ru/ninja/
    path = request.path
    if host == settings.CRT_NINJA_DOMAIN:
        if path.startswith('/ninja/'):
            path = path[len('/ninja'):]
        elif path == '/ninja':
            path = '/'

    # Далее исходная версия (за исключением использования новой переменной path вместо request.path)
    schema = 'https' if request.is_secure() else 'http'
    url = ('%s://%s%s' % (schema, host, path)).encode('utf-8')
    params = request.GET
    if add_params or del_params:
        params = request.GET.copy()
        params.update(add_params or {})
        for p in del_params or []:
            del params[p]
    if params:
        url += b'?' + params.urlencode().encode('utf-8')
    return url


class FixedAuthRequiredMiddleware(middleware.YandexAuthRequiredMiddleware):
    """
    Аналог YandexAuthRequiredMiddleware, которая умеет в редиректы и обработку
    отсутствия или невалидности одной из кук
    """

    @staticmethod
    def _set_authentication_descriptors(request):
        # Дефолтная мидлварь не требует залогиненности, а та, которая требует,
        # не проставляет все необходимые атрибуты
        request.__class__.yauser = YandexUserDescriptor()
        request.__class__.client_application = ApplicationDescriptor()
        if get_setting(['YAUSER_ADMIN_LOGIN', 'YAUTH_USE_NATIVE_USER']):
            request.__class__.user = DjangoUserDescriptor()

    def process_request(self, request):
        self._set_authentication_descriptors(request)

        is_authenticated = False
        is_oauth = False
        try:
            if request.yauser.is_authenticated():
                is_authenticated = True
                if settings.YAUTH_ANTI_REPEAT_PARAM in request.GET:
                    # Убираем YAUTH_ANTI_REPEAT_PARAM, чтобы не ломать ссылку Выйти.
                    return http.HttpResponseRedirect(
                        get_retpath(request, del_params=[settings.YAUTH_ANTI_REPEAT_PARAM])
                    )
            else:
                is_oauth = request.yauser.oauth
        except (exceptions.TwoCookiesRequired, exceptions.AuthRequired):
            pass

        if not is_authenticated:
            if request.method == 'GET' and not is_oauth:
                if settings.YAUTH_ANTI_REPEAT_PARAM in request.GET:
                    # чтобы не редиректить бесконечно на Паспорт
                    return http.HttpResponseForbidden('Authorization failed' + debug_msg(request)
                                                      if settings.DEBUG
                                                      else '')

                retpath = get_retpath(request, add_params={settings.YAUTH_ANTI_REPEAT_PARAM: 1})
                passport_url = get_passport_url(
                    url_type='create',
                    yauth_type=get_yauth_type(request),
                    request=request,
                    retpath=retpath,
                )

                return http.HttpResponseRedirect(passport_url)
            elif request.yauser.blackbox_result is None:
                return http.HttpResponse('Authentication required', status=401)
            else:
                # Случается в редких случаях, когда юзер до-о-олго сидела на
                # странице с открытой формой, и авторизация протухла.
                # Спасти данные мы уже не можем.
                return http.HttpResponseForbidden('Authorization failed')


class AuthenticationWhitelistedMiddleware(FixedAuthRequiredMiddleware):
    # Пути, в которых действует "фронтовая" аутентификация с редиректами на Паспорт
    NORMALLY_AUTHENTICATED_PATHS = [
        '^/ninja$',
    ]
    # Пути, в которых вообще не проверяется аутентифицированность
    UNAUTHENTICATED_PATHS = [
        '^/api/frontend/userdata$',
        '^/ping$',
        '^/migration$',
        '^/api/hosts-to-approve.xml$',
        '^/api/hosts/',  # это только префикс
        '^/monitorings/',  # тоже префикс
        '^/idm/',  # и снова префикс
    ]

    @classmethod
    def is_in_path_group(cls, path, path_group):
        prepared_path = '^{}$'.format(path.rstrip('/'))
        for prefix in path_group:
            if prefix in prepared_path:
                return True
        return False

    def process_request(self, request):
        if self.is_in_path_group(request.path, self.NORMALLY_AUTHENTICATED_PATHS):
            return super(AuthenticationWhitelistedMiddleware, self).process_request(request)

        # Возможно уже смогли достать пользователя из клиентского сертификата
        try:
            if getattr(request, 'user', None) and request.user.is_authenticated():
                return
        except exceptions.AuthException:
            pass

        # Проставляем пользователя, но редиректы игнорируем
        super(AuthenticationWhitelistedMiddleware, self).process_request(request)

        if self.is_in_path_group(request.path, self.UNAUTHENTICATED_PATHS):
            return

        authenticated = False
        try:
            if request.user.is_authenticated():
                authenticated = True
        except exceptions.TwoCookiesRequired:  # no sessionid2 or invalid (expired) cookies
            pass

        if not authenticated:
            return NotAuthenticatedResponse(b'User is not authenticated')


class QloudSecretHeaderMiddleware(object):
    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        # если заголовки не совпадают и ручка не из whitelist
        if (settings.CRT_BALANCER_SECRET_HEADER != request.META.get('HTTP_X_CRT_BALANCER_SECRET_HEADER') and
                not any(request.path.startswith(whitelist_url)
                        for whitelist_url in settings.CRT_IGNORE_SECRET_HEADER_URLS)):
            with log_context(method=request.method, path=request.path):
                log.warning('Invalid HTTP_X_CRT_BALANCER_SECRET_HEADER passed')
            return NotAuthenticatedResponse(b'invalid HTTP_X_CRT_BALANCER_SECRET_HEADER')

        return self.get_response(request)
