import datetime
import logging
from collections import defaultdict

from django.conf import settings
from django.utils import timezone
from django.utils.encoding import force_text

from intranet.crt.constants import CERT_STATUS
from intranet.crt.core.ca import external_ca
from intranet.crt.core.ca.base import BaseCA
from intranet.crt.core.ca.exceptions import RetryCaException, CaError
from intranet.crt.core.ca.zeep_client import ZeepClient
from intranet.crt.core.notifications import notify
from intranet.crt.utils.ssl import PemCertificateRequest, PemCertificate

log = logging.getLogger(__name__)


class CertumZeepClient(ZeepClient):
    VALID_PERIOD = 182  # Cертификаты CertumCA выдаются сроком на полгода (CERTOR-1551)
    SHA2_PRODUCT_ID = 971  # Trusted MultiDomain SSL 200 Domains (1 year) [API PDF]
    EV_PRODUCT_ID = 981  # Premium EV MultiDomain SSL 200 Domains (1 year)
    IGNORE_ERRORS = {
        1185,  # Domain Has already been verified
    }
    SUCCESS_REVOKE_ERRORS = {
        1033,  # Order ID does not exist
        1151,  # Order has already cancelled
        1145,  # Certificate has already been revoked
    }
    NAMESPACE = 'ns0'

    def get_errors(self, response):
        header = response.responseHeader if hasattr(response, 'responseHeader') else response

        if header.successCode == 0:
            return {}

        if not hasattr(header, 'errors'):
            raise RetryCaException('No errors block in response', response)

        errors = {int(error.errorCode) for error in header.errors.Error} - self.IGNORE_ERRORS

        # https://st.yandex-team.ru/CERTOR-178
        if 1 in errors:
            raise RetryCaException(errors, response)

        return errors

    def _call(self, method, *args, **kwargs):
        kwargs['requestHeader'] = self.create_request_header()
        return super(CertumZeepClient, self).call(method, *args, **kwargs)

    def call(self, method, *args, **kwargs):
        response = self._call(method, *args, **kwargs)

        errors = self.get_errors(response)
        if errors:
            raise CaError(errors, response)

        return response

    def revoke_call(self, method, *args, **kwargs):
        response = self._call(method, *args, **kwargs)

        errors = self.get_errors(response)
        if errors and not (errors & self.SUCCESS_REVOKE_ERRORS):
            raise CaError(errors, response)

        return response

    def create_request_header(self):
        authToken = self.factory.authToken(
            userName=self.username,
            password=self.password,
        )
        header = self.factory.requestHeader(authToken=authToken)
        return header

    def create_requestor_info(self):
        info = self.factory.requestorInfo()
        info.email = settings.CRT_PROD_CERT_CONTACTS['email']
        info.firstName = settings.CRT_PROD_CERT_CONTACTS['first_name']
        info.lastName = settings.CRT_PROD_CERT_CONTACTS['last_name']
        info.phone = settings.CRT_PROD_CERT_CONTACTS['phone']
        return info

    def create_order_option(self):
        option = self.factory.orderOption()
        option.orderStatus = True
        option.orderDetails = True
        option.certificateDetails = True
        return option

    def create_order_parameters(self, csr, is_extended_validation, desired_ttl_days):
        if desired_ttl_days is None:
            ttl = self.VALID_PERIOD
        else:
            ttl = min(desired_ttl_days, self.VALID_PERIOD)
        not_after = timezone.now().date() + datetime.timedelta(days=ttl)

        params = self.factory.orderParameters()
        params.CSR = csr
        params.customer = 'pki-customer'
        params.language = 'RU'
        params.productCode = self.EV_PRODUCT_ID if is_extended_validation else self.SHA2_PRODUCT_ID
        params.shortenedValidityPeriod = '{0:%Y-%m-%d}'.format(not_after)
        if PemCertificateRequest(csr).is_ecc:
            params.hashAlgorithm = 'ECC-SHA256'
        return params

    def create_san_entries(self, domains):
        entries = self.factory.sanEntries()

        for host in domains:
            entry = self.factory.sanEntry()
            entry.DNSName = host

            entries.SANEntry.append(entry)

        return entries

    def create_fqdns(self, domains):
        fqdns = self.factory.FQDNs()
        fqdns.FQDN = domains
        return fqdns

    def create_approver(self):
        approver = self.factory.sanApprover()
        approver.approverMethod = 'DNS'
        approver.approverEmail = 'admin@yandex.ru'
        approver.verificationNotificationEnabled = False

        return approver

    def order(self, csr, domains, is_extended_validation, desired_ttl_days):
        order_parameters = {
            'requestorInfo': self.create_requestor_info(),
            'orderParameters': self.create_order_parameters(
                csr,
                is_extended_validation,
                desired_ttl_days,
            ),
            'SANEntries': self.create_san_entries(domains),
            'SANApprover': self.create_approver(),
        }
        self.call('validateOrderParameters', **order_parameters)
        return self.call('quickOrder', **order_parameters)

    def get_order(self, request_id):
        order_option = self.create_order_option()
        response = self.call('getOrderByOrderID', orderID=request_id, orderOption=order_option)

        return response.orders.Order[0]

    def revoke_certificate(self, serial_number):
        parameters = self.factory.revokeCertificateParameters()
        parameters.serialNumber = serial_number

        self.revoke_call('revokeCertificate', revokeCertificateParameters=parameters)

    def cancel_order(self, request_id):
        parameters = self.factory.cancelParameters()
        parameters.orderID = request_id

        self.revoke_call('cancelOrder', cancelParameters=parameters)

    def verify_domain(self, code):
        self.call('performSanVerification', code=code)

    def get_verification_state(self, request_id):
        return self.call('getSanVerificationState', orderID=request_id)

    def get_verifications(self, order_response) -> list[external_ca.VerificationDomainInfo]:
        verification = getattr(order_response, 'SANVerification', {})

        if verification.approverMethod != 'DNS':
            log.warning('Мethod DNS was excpected, but got %s', verification.approverMethod)
            return []

        # verification.FQDN – это suds.sax.text.Text, унаследованный от unicode,
        # и в целом с ним почти всё хорошо (кроме сломанного repr), но кажется с ним не умеет
        # нормально работать драйвер MySQL
        # Поэтому после force_text нужно ещё раз вызвать unicode, чтобы привести его к нужному типу
        # Без этого база сваливается в трейс как в CERTOR-652
        result = [
            external_ca.VerificationDomainInfo(str(force_text(fqdn)), verification.code)
            for fqdn in verification.FQDNs.FQDN
        ]
        return result


class CertumProductionCA(BaseCA):
    chain_filename = 'YandexCA.pem'

    def __init__(self, wsdl_url, service_url, username, password, supported_types):
        super(CertumProductionCA, self).__init__()
        self.wsdl_url = wsdl_url
        self.service_url = service_url
        self.supported_types = supported_types
        self.credentials = {
            'username': username,
            'password': password,
        }

    def get_client(self):
        return CertumZeepClient.create(self.wsdl_url, **self.credentials)

    @staticmethod
    def get_autovalidate_domain_names():
        from intranet.crt.core.models import HostToApprove
        query = HostToApprove.objects.filter(auto_managed=True, managed_dns=True)
        return set(query.values_list('host', flat=True))

    def verify_domain(self, verification_code):
        log.info('Trying to validate code %s in Certum', str(verification_code))
        client = self.get_client()
        client.verify_domain(verification_code.code)
        log.info('Verification request was sent')

    def get_validation_status(self, validation_code):
        from intranet.crt.core.models import HOST_VALIDATION_CODE_STATUS
        verifications: dict[str: dict] = self._get_verification_state_by_hosts(validation_code.certificate)
        zones = validation_code.certificate.hosts_to_approve.values_list('host', flat=True)
        zone_to_verifications = defaultdict(list)
        for host, ver in verifications.items():
            cur_zone = ''
            for zone in zones:
                # we need the most precize zone
                if host.endswith(zone) and len(zone) > len(cur_zone):
                    cur_zone = zone
            if cur_zone:
                zone_to_verifications[cur_zone].append(ver)
        verifications = zone_to_verifications[validation_code.host.host]

        if not verifications:
            log.error('Code %s is not found in certificate validation state', validation_code.code)
            return HOST_VALIDATION_CODE_STATUS.error

        # status can be unchanged - this means we need to wait
        status = validation_code.status
        for verification in verifications:
            if verification['state'] == 'VERIFIED':
                status = HOST_VALIDATION_CODE_STATUS.validated
            elif verification['state'] == 'FAILED':
                if verification['error_code'] == 'ALREADY_VERIFIED':
                    log.info('Code %s has been validated already', validation_code.code)
                    status = HOST_VALIDATION_CODE_STATUS.validated
                # статусы про DNS просто пропускаем - ждём дальше
                elif verification['error_code'] not in ['DNS_NO_RECORDS', 'DNS_NO_PROPER_RECORDS']:
                    log.warning(
                        'Code %s is not validated. Reason: %s',
                        validation_code.code,
                        verification['error_code'],
                    )
                    return HOST_VALIDATION_CODE_STATUS.error
        return status

    def _get_verification_state_by_hosts(self, cert):
        """
        Возвращает статусы верификации хостов для конкретного заказа.
        Возможные статусы: REQUIRED, FAILED, VERIFIED
        Возможные error_code при статусе:
            - ALREADY_VERIFIED - verification has already been successfully performer
            - LINK_EXPIRED – verification link has expired
            - OTHER_ERROR – unknown reason of error
            - FILE_INVALID_CONTENT – invalid content of the verification file
            - FILE_CONNECTION_ERROR – unable to find the verification file
            - FILE_HTTP_ERROR - unable to find the verification file
            - DNS_NO_RECORDS – no TXT records on the DNS server
            - DNS_NO_PROPER_RECORDS - – no properly TXT records on the DNS server
        """
        client = self.get_client()
        response = client.get_verification_state(cert.request_id)
        result = {}
        for verification in response.sanVerifications.sanVerification:
            result[verification.FQDN] = {
                'state': verification.manualVerification.state,
                'error_code': verification.manualVerification.info,
            }
        return result

    def _issue(self, cert):
        try:
            if cert.request_id is None:
                cert.request_id = self._send_request(cert)
                cert.status = CERT_STATUS.VALIDATION
                cert.save()

                raise RetryCaException('Accept request ID {}'.format(cert.request_id))

            elif cert.certificate is None:
                certificate = self._get_certificate(cert.request_id)
                if certificate is None:
                    raise RetryCaException('Certificate still can not be issued')
                pem_certificate = PemCertificate(certificate)
                now = timezone.now()
                if pem_certificate.not_before > now:
                    raise RetryCaException(
                        'Certificate can be issued, but not_before (%s) is greater now (%s)'
                        % (pem_certificate.not_before, now)
                    )

                # Если сертифкат заказан по csr, то отправляем письмо
                # иначе отправим письмо после записи ключа в секретницу
                if cert.requested_by_csr:
                    notify(
                        cert.user.email,
                        'ssl-certificate-was-issued',
                        subject='SSL сертификат готов',
                        certificate=cert,
                        hosts=[host.hostname for host in cert.hosts.all()],
                    )
                return certificate
        except CaError as error:
            notify(
                cert.user.email,
                'ssl-certificate-was-not-issued',
                subject='SSL сертификат не был выдан из-за ошибки',
                certificate=cert,
                hosts=[host.hostname for host in cert.hosts.all()],
                error=force_text(error),
            )
            raise

    def revoke(self, cert):
        if not cert.request_id:
            return

        client = self.get_client()
        if cert.serial_number:
            client.revoke_certificate(cert.serial_number)
        else:
            log.info('There is no serial number yet, so we just cancel the order')
            client.cancel_order(cert.request_id)

    def _send_request(self, cert):
        log.info('Sending a certificate request')

        csr = cert.request
        is_extended_validation = cert.extended_validation
        domains = cert.get_sans()

        client = self.get_client()
        response = client.order(csr, domains, is_extended_validation, cert.desired_ttl_days)

        verifications = client.get_verifications(response)
        external_ca.save_cert_verification_info(cert, verifications)

        log.info('Order %s was placed', response.orderID)
        return response.orderID

    def _get_certificate(self, request_id):
        log.info('Trying to retrive certificate for request_id %s', request_id)

        client = self.get_client()
        order = client.get_order(request_id)
        log.info('Got order status %s for request_id %s', order.orderStatus.orderStatus, request_id)

        try:
            certificate = order.certificateDetails.X509Cert
        except AttributeError:
            return None

        begin = '-----BEGIN CERTIFICATE-----\n'
        end = '\n-----END CERTIFICATE-----'

        if not certificate.startswith(begin):
            certificate = begin + certificate + end

        return certificate


class CertumTestCA(CertumProductionCA):
    chain_filename = 'YandexCA.pem'
