import dataclasses
import datetime
import logging
import typing as tp

from django.conf import settings
from django.utils import timezone

from intranet.crt.utils.domain import domain_is_toplevel


log = logging.getLogger(__name__)


@dataclasses.dataclass
class VerificationDomainInfo:
    fqdn: str
    code: tp.Optional[str] = None
    globalsign_domain_id: tp.Optional[str] = None


def save_cert_verification_info(cert, verifications: list[VerificationDomainInfo]):
    for verification in verifications:
        log.info('External CA wants to verify domain: %s', verification)

        if domain_is_toplevel(verification.fqdn):
            log.info('Host %s was placed to the queue for verification', verification.fqdn)
            _save_domain_verification_info(cert, verification)
        else:
            log.info('host %s was skipped as a non top level', verification.fqdn)


def _save_domain_verification_info(certificate, verification: VerificationDomainInfo):
    """Добавляет указанный хост в очередь на валидацию."""

    # TODO(rocco66): fix circular deps
    from intranet.crt.core.models import HostToApprove, HostValidationCode
    approve, created = HostToApprove.objects.get_or_create(host=verification.fqdn)
    approve.certificates.add(certificate)
    if verification.globalsign_domain_id:
        approve.globalsign_domain_id = verification.globalsign_domain_id
    approve.save()

    if verification.code:
        HostValidationCode.objects.create(
            host=approve,
            certificate=certificate,
            code=verification.code,
            expire_at=timezone.now() + datetime.timedelta(0, settings.TXT_RECORD_TTL),
        )
    approve.history.create(action='added', validation_code=verification.code, certificate=certificate)
    log.info('Host now waiting for validation with code %s', verification)


