import collections
import datetime
import functools
import logging

from django.conf import settings

from intranet.crt.constants import CERT_STATUS, HOST_VALIDATION_CODE_STATUS
from intranet.crt.core.ca import external_ca
from intranet.crt.core.ca.base import BaseCA
from intranet.crt.core.ca.exceptions import CaError, RetryCaException
from intranet.crt.core.ca.zeep_client import ZeepClient
from intranet.crt.utils.domain import get_domain_levels, domain_is_toplevel

log = logging.getLogger(__name__)


class GlobalSignZeepClient(ZeepClient):
    NAMESPACE = 'ns0'


class GlobalSignProductionCA(BaseCA):
    # Константы взяты из документации. Подробности про настройки можно найти в документации
    # по ctrl+f <название главы>. Документацию можно посмотреть тут: https://proxy.sandbox.yandex-team.ru/3183976310

    chain_filename = 'globalsign-ov.pem'  # TODO: для EV может быть другая цепочка
    ecc_chain_filename = 'globalsign-ecc-ov.pem'

    PERMISSION_REQUIRED = 'core.can_issue_globalsign'
    ISSUE_TIMEOUT_IN_DAYS = 10

    # 6.10 Subject Alternative Names (SANs)
    SAN_UC = 1
    SAN_SUBDOMAIN = 2
    SAN_GIP = 3
    SAN_INTERNAL = 4
    SAN_FQDN = 7
    SAN_WILDCARD_FQDN = 13

    # 6.12 Validity Period
    DAYS_THRESHOLDS = collections.OrderedDict((
        (184, 6),
        (366, 12),
        (731, 24),
        (1096, 36),
        (1461, 48),
        (1826, 60),
    ))

    # 8.1 Order/Certificate Status
    CERTIFICATE_STATUS_INITIAL = '1'
    CERTIFICATE_STATUS_CHECKING = '2'
    CERTIFICATE_STATUS_CANCELLED = '3'
    CERTIFICATE_STATUS_ISSUED = '4'
    CERTIFICATE_STATUS_CANCELLED_ISSUED = '5'
    CERTIFICATE_STATUS_REVOCATION = '6'
    CERTIFICATE_STATUS_REVOKED = '7'

    # 8.3 MSSL Domain Status
    DOMAIN_STATUS_INITIAL = '1'
    DOMAIN_STATUS_VETTING = '2'
    DOMAIN_STATUS_AVAILABLE = '3'
    DOMAIN_STATUS_SUSPENDED = '5'
    DOMAIN_STATUS_REJECTED = '6'
    DOMAIN_STATUSES = {
        DOMAIN_STATUS_INITIAL: 'INITIAL / Vetting in Progress',
        DOMAIN_STATUS_VETTING: 'Vetting in Progress',
        DOMAIN_STATUS_AVAILABLE: 'Vetting Completed / Available',
        DOMAIN_STATUS_SUSPENDED: 'Cancelled / Suspended',
        DOMAIN_STATUS_REJECTED: 'Domain Rejected',
    }

    # 8.5.2 Client Error Codes
    ERROR_WRONG_STATUS_FOR_OPERATION = -9938
    ERROR_ORDER_NOT_FOUND = -9916
    ERROR_DOMAIN_NAME_ALREADY_EXISTS_FOR_THE_MSSL_PROFILE_ID = -9406
    ERROR_INTERNAL_SYSTEM_ERROR = -1

    def __init__(self, hostname, username, password, profile_id, supported_types):
        super(GlobalSignProductionCA, self).__init__()
        self.hostname = hostname
        self.username = username
        self.password = password
        self.profile_id = profile_id
        self.supported_types = supported_types

    @functools.cached_property
    def mssl_functions_client(self):
        # TODO(rocco66): migration to API last version
        return GlobalSignZeepClient.create(
            f'https://{self.hostname}/kb/ws/v1/ManagedSSLService?wsdl', self.username, self.password,
        )

    @functools.cached_property
    def query_client(self):
        return GlobalSignZeepClient.create(
            f'https://{self.hostname}/kb/ws/v1/GASService?wsdl', self.username, self.password,
        )

    @staticmethod
    def get_autovalidate_domain_names():
        from intranet.crt.core.models import HostToApprove
        query = HostToApprove.objects.filter(auto_managed=True, managed_dns=True)
        return set(query.values_list('host', flat=True))

    @staticmethod
    def _get_unknown_top_level_domains(cert) -> set[str]:
        from intranet.crt.core.models import HostToApprove
        target_domains = set()
        for host in cert.hosts.all():
            if top_level_domain := next((d for d in get_domain_levels(host.hostname) if domain_is_toplevel(d)), None):
                target_domains.add(top_level_domain)
        known_domains = set(
            HostToApprove.objects
                .filter(host__in=list(target_domains))
                .exclude(globalsign_domain_id__isnull=True)
                .values_list("host", flat=True)
        )
        return set(target_domains) - known_domains

    def _issue(self, cert):
        certificate = None
        unknown_domain_ids = self._get_unknown_top_level_domains(cert)
        if unknown_domain_ids and not cert.extended_validation:
            # NOTE(rocco66): extended_validation does not supported right now
            self._add_domains_to_profile(cert, unknown_domain_ids)
        elif (
            cert.request_id is None
            and not cert.validation_codes.exclude(status=HOST_VALIDATION_CODE_STATUS.validated).exists()
        ):
            certificate = self._issue_certificate(cert)
        elif cert.request_id and cert.certificate is None:
            certificate = self._fetch_certificate(cert)

        if certificate is None:
            raise RetryCaException('Certificate still not issued')

        return certificate

    def revoke(self, cert):
        if not cert.request_id:
            return

        status = self._get_order_status(cert)

        if status in (self.CERTIFICATE_STATUS_INITIAL, self.CERTIFICATE_STATUS_CHECKING):
            operation = 'CANCEL'
        elif status in (self.CERTIFICATE_STATUS_ISSUED, self.CERTIFICATE_STATUS_CANCELLED_ISSUED):
            operation = 'REVOKE'
        else:
            return

        client = self.mssl_functions_client
        request = client.factory.BmV1ModifyMsslOrderRequest(
            OrderRequestHeader=self._make_order_header(client.factory),
            OrderID=cert.request_id,
            ModifyOrderOperation=operation
        )

        response = client.call('ModifyMSSLOrder', request)
        self._assert_successful_response(response, allow_errors={
            self.ERROR_ORDER_NOT_FOUND,
            self.ERROR_WRONG_STATUS_FOR_OPERATION,
        })

    def verify_domain(self, validation_code):
        client = self.mssl_functions_client
        request = client.factory.BmV1VerifyMsslDomainRequest(
            OrderRequestHeader=self._make_order_header(client.factory),
            DomainID=validation_code.host.globalsign_domain_id,
            TagLocation=validation_code.host.host,
            VettingType="DNS",
        )
        response = client.call('VerifyMsslDomain', request)
        self._assert_successful_response(response)

    def get_validation_status(self, validation_code) -> HOST_VALIDATION_CODE_STATUS:
        client = self.mssl_functions_client
        request = client.factory.BmV1GetMsslDomainListRequest(
            QueryRequestHeader=self._make_query_header(client.factory),
            MSSLProfileID=self.profile_id,
            MSSLDomainID=validation_code.host.globalsign_domain_id,
        )
        response = client.call('GetMSSLDomains', request)
        self._assert_successful_response(response)

        if not response.SearchMsslDomainDetails:
            raise RuntimeError(f"unknown GetMSSLDomains response structure {response}")

        if target_domain_info := next(iter(response.SearchMsslDomainDetails.SearchMsslDomainDetail), None):
            text_status = self.DOMAIN_STATUSES[target_domain_info.MSSLDomainStatus]
            log.info('Found domain %s with status %s', target_domain_info.MSSLDomainName, text_status)
            if target_domain_info.MSSLDomainStatus == self.DOMAIN_STATUS_AVAILABLE:
                return HOST_VALIDATION_CODE_STATUS.validated
            else:
                return HOST_VALIDATION_CODE_STATUS.validation
        else:
            log.info('Domain "%s" was not found in GetMsslDomainList response', target_domain_info.MSSLDomainName)
            return HOST_VALIDATION_CODE_STATUS.error

    def _get_order_status(self, cert):
        client = self.query_client
        request = self._make_fetch_request(cert, client.factory, return_body=False)
        response = client.call('GetOrderByOrderID', request)
        self._assert_successful_response(response)
        return str(response.OrderDetail.OrderInfo.OrderStatus)

    def _add_domains_to_profile(self, cert, target_domains: set[str]):
        verification_infos = []
        for domain in target_domains:
            if cert.extended_validation:
                # NOTE(rocco66): it does not worked right now
                raise RuntimeError("EV does not supported for globalsign right now")
            else:
                client = self.mssl_functions_client
                factory = client.factory
                request = factory.BmV1AddDomainToProfileRequest(
                    OrderRequestHeader=self._make_order_header(factory),
                    MSSLProfileID=self.profile_id,
                    DomainName=domain,
                    DomainID="",
                    ApproverEmail="",
                    VettingLevel="OV",
                    VettingType="DNS",
                    ContactInfo=self._make_contacts(factory),
                )
                response = client.call('AddDomainToProfile', request)
                if self._assert_successful_response(response, allow_errors={
                    self.ERROR_DOMAIN_NAME_ALREADY_EXISTS_FOR_THE_MSSL_PROFILE_ID,
                }):
                    # TODO(rocco66): domain info fetching
                    raise RetryCaException('Domain wa already added (race?)')

                verification_code = str(response.DnsTXT)

            verification_infos.append(external_ca.VerificationDomainInfo(
                domain, verification_code, str(response.MSSLDomainID),
            ))

        external_ca.save_cert_verification_info(cert, verification_infos)
        cert.status = CERT_STATUS.VALIDATION
        cert.save()

    def _issue_certificate(self, cert):
        client = self.mssl_functions_client
        request = self._make_issue_request(cert, client.factory)
        response = client.call('PVOrder', request)
        self._assert_successful_response(response)

        cert.request_id = response.OrderID
        cert.status = CERT_STATUS.VALIDATION
        cert.save()
        if response.PVOrderDetail and response.PVOrderDetail.Fulfillment:
            certificate = response.PVOrderDetail.Fulfillment.ServerCertificate.X509Cert
            return certificate

    def _make_validity_period(self, cert, factory):
        actual_days = settings.CRT_GLOBALSIGN_CERTS_MAX_TTL

        if cert.desired_ttl_days is not None:
            actual_days = min(cert.desired_ttl_days, actual_days)

        # Months может быть 6/12/24/36/48, нам нужен минимальный
        actual_months = None
        for days, months in self.DAYS_THRESHOLDS.items():
            if actual_days <= days:
                actual_months = months
                break

        # Реальный EndDate будет в последнюю секунду дня перед not_after
        not_after = datetime.datetime.today() + datetime.timedelta(days=actual_days)

        period = factory.ValidityPeriod()
        period.NotAfter = not_after.strftime('%Y-%m-%d')
        period.Months = actual_months
        return period

    def _assert_successful_response(self, response, allow_errors=None) -> list[int]:
        header_names = ('QueryResponseHeader', 'OrderResponseHeader')
        header = None
        for header_name in header_names:
            if hasattr(response, header_name):
                header = getattr(response, header_name)
                break

        errors = []

        if header:
            if header.SuccessCode != 0:
                for error in header.Errors.Error:
                    errors.append(int(error.ErrorCode))
        else:
            errors.append(0)

        if errors and (not allow_errors or set(errors) - set(allow_errors)):
            raise CaError(errors, response)
        return errors

    def _make_contacts(self, factory):
        return factory.ContactInfo(
            FirstName=settings.CRT_PROD_CERT_CONTACTS['first_name'],
            LastName=settings.CRT_PROD_CERT_CONTACTS['last_name'],
            Phone=settings.CRT_PROD_CERT_CONTACTS['phone'],
            Email=settings.CRT_PROD_CERT_CONTACTS['email'],
        )

    def _make_options(self, option_names, factory):
        options = factory.Options()
        for optname in option_names:
            options.Option.append(factory.Option(
                OptionName=optname,
                OptionValue=True,
            ))
        return options

    def _make_issue_request(self, cert, factory):
        request = factory.BmV1PvOrderRequest(
            ContactInfo=self._make_contacts(factory),
            MSSLDomainID=self._detect_domain_id(cert),
            MSSLProfileID=self.profile_id,
            OrderRequestHeader=self._make_order_header(factory),
            OrderRequestParameter=factory.OrderRequestParameter(
                BaseOption=None,
                CSR=cert.request,
                Licenses=1,
                Options=None,
                OrderKind='new',
                ProductCode='PEV_SHA2' if cert.extended_validation else 'PV_SHA2',
                ValidityPeriod=self._make_validity_period(cert, factory),
            ),
            PVSealInfo=None,
            SANEntries=None,
        )

        options = []

        if cert.desired_ttl_days:
            # Чтобы можно было указывать NotAfter и NotBefore
            options.append('VPC')

        if cert.hosts.count() > 1:
            request.SANEntries = self._make_sans(cert, factory)
            if request.SANEntries:
                options.append('SAN')

        params = request.OrderRequestParameter
        params.Options = self._make_options(options, factory)
        if cert.wildcards_count():
            params.BaseOption = 'wildcard'

        return request

    def _detect_domain_id(self, cert):
        domain_names = get_domain_levels(cert.common_name)
        from intranet.crt.core.models import HostToApprove

        query = HostToApprove.objects.filter(host__in=domain_names)\
            .exclude(globalsign_domain_id=None)

        host = query.first()
        if host:
            return host.globalsign_domain_id

        raise RuntimeError('Domain in common name is not pre-validated.')

    def _make_sans(self, cert, factory):
        sans = None
        fqdns = cert.get_sans2()
        if fqdns:
            sans = factory.SANEntries()
            for fqdn in fqdns:
                san_type = self.SAN_WILDCARD_FQDN if fqdn.startswith('*.') else self.SAN_FQDN
                sans.SANEntry.append(factory.SANEntry(
                    SubjectAltName=fqdn,
                    SANOptionType=san_type,
                ))
        return sans

    def _make_auth_token(self, factory):
        return factory.AuthToken(
            UserName=self.username,
            Password=self.password,
        )

    def _make_order_header(self, factory):
        return factory.OrderRequestHeader(
            AuthToken=self._make_auth_token(factory)
        )

    def _make_query_header(self, factory):
        return factory.QueryRequestHeader(
            AuthToken=self._make_auth_token(factory)
        )

    def _make_fetch_request(self, cert, factory, return_body=True):
        request = factory.QbV1GetOrderByOrderIdRequest(
            QueryRequestHeader=self._make_query_header(factory),
            OrderID=cert.request_id,
        )

        if return_body:
            request.OrderQueryOption = factory.OrderQueryOption(
                ReturnFulfillment=True,
            )

        return request

    def _fetch_certificate(self, cert):
        client = self.query_client

        request = self._make_fetch_request(cert, client.factory)
        response = client.call('GetOrderByOrderID', request)

        self._assert_successful_response(response)

        if response.OrderDetail.OrderInfo.OrderStatus == self.CERTIFICATE_STATUS_ISSUED:
            return response.OrderDetail.Fulfillment.ServerCertificate.X509Cert

    def add_domain_for_vetting(self, domain, level='OV'):
        # TODO (ipeterov): можно начать использовать,
        # чтобы автоматически ставить хост в очередь на аппрув
        client = self.mssl_functions_client
        request = client.factory.BmV1AddMsslDomainRequest(
            DomainName=domain,
            MSSLProfileID=self.profile_id,
            OrderRequestHeader=self._make_order_header(client.factory),
            VettingLevel=level,
        )
        response = client.call('AddMSSLDomain', request)
        self._assert_successful_response(response)
        return response


class GlobalSignTestCA(GlobalSignProductionCA):
    chain_filename = 'globalsign-ov-test.pem'  # TODO: для EV может быть другая цепочка
