from html import parser
import logging
import re
import time
import urllib.parse

import sys

from django.conf import settings
from django.utils import six
from rest_framework import status

from intranet.crt.core.ca.base import BaseCA
from intranet.crt.core.ca import exceptions
from intranet.crt.exceptions import CrtRequestError
from intranet.crt.utils.http import CrtSession
from intranet.crt.utils.domain import get_domain_levels

log = logging.getLogger(__name__)


class HTMLToTextParser(parser.HTMLParser):
    def __init__(self):
        super().__init__()
        self.words = []

    def handle_data(self, data):
        data = data.strip()
        if data:
            self.words.append(data)

    def get_text(self):
        return ' '.join(self.words)


def parse_ca_response_to_text(response):
    try:
        parser = HTMLToTextParser()
        parser.feed(response)
        return parser.get_text()
    except:
        return str(response)


class InternalCaSession(CrtSession):
    DEFAULT_TIMEOUT = (settings.CRT_INTERNAL_CA_CONNECT_TIMEOUT, settings.CRT_INTERNAL_CA_READ_TIMEOUT)

    def __init__(self, username, password):
        super(InternalCaSession, self).__init__()

        self.set_basic_auth(username, password)

    def request(self, method, url, **kwargs):
        try:
            return super(InternalCaSession, self).request(method, url, **kwargs)
        except CrtRequestError as error:
            six.reraise(exceptions.CaError, error, sys.exc_info()[2])


class InternalCA(BaseCA):
    CERT_URL_PATTERN = re.compile(r'certnew.cer\?ReqID=\d+&amp;Enc=b64')
    IS_EXTERNAL = False
    IS_ASYNC = False

    chain_filename = settings.YANDEX_CHAIN

    def __init__(self, username, password, request_url, revoke_url, supported_types):
        super(InternalCA, self).__init__()

        self.request_url = request_url
        self.revoke_url = revoke_url
        self.supported_types = supported_types

        self.session = InternalCaSession(username, password)

    def cert_request(self, cert):
        issue_url = urllib.parse.urljoin(self.request_url, '/certsrv/certfnsh.asp')

        post_data = {
            'Mode': 'newreq',
            'CertAttrib': 'CertificateTemplate:{0}'.format(cert.used_template),
            'CertRequest': cert.request,
        }
        response = self.session.post(issue_url, data=post_data)
        self.raise_non_200(response)

        match = self.CERT_URL_PATTERN.search(response.text)
        if match is None:
            timestamp = time.asctime()
            answer_text = parse_ca_response_to_text(response.text)
            raise exceptions.CaError(
                f'Invalid answer from Internal CA at {timestamp}. Answer is: {answer_text}'
            )

        return urllib.parse.urljoin(response.url, match.group(0).replace('&amp;', '&'))

    def revoke_action(self, cert, type):
        """Описание ручки на вики отзыва сертификатов:
        https://wiki.yandex-team.ru/DljaAdminov/WinAdmin/RevokeWebService
        """
        post_data = {
            'Serial': cert.serial_number,
            'Reason': type,
        }
        response = self.session.post(self.revoke_url, data=post_data)
        self.raise_non_200(response)

    def get_certificate(self, url):
        response = self.session.get(url)
        self.raise_non_200(response)
        return response.content

    def _issue(self, cert):
        used_template = cert.controller.get_internal_ca_template()
        if used_template is None:
            raise exceptions.ValidationCaError('Cannot find internal CA template')

        cert.used_template = used_template
        cert_url = self.cert_request(cert)
        return self.get_certificate(cert_url)

    def revoke(self, cert):
        if cert.is_imported or cert.certificate:
            self.revoke_action(cert, 'CRL_REASON_CESSATION_OF_OPERATION')
            log.info('Revoking certificate %s', cert.serial_number)
        else:
            log.info('Can\'t revoke certificate %s', cert.serial_number)

    def hold(self, cert):
        self.revoke_action(cert, 'CRL_REASON_CERTIFICATE_HOLD')

    def raise_non_200(self, response):
        if response.status_code == status.HTTP_429_TOO_MANY_REQUESTS:
            raise exceptions.Ca429Error(response.content, retry_after=response.headers.get('Retry-After'))
        elif response.status_code != status.HTTP_200_OK:
            raise exceptions.CaError(
                'Internal CA returned wrong status code: {0}, answer: {1}'.format(response.status_code, response.content)
            )

    def unhold(self, cert):
        self.revoke_action(cert, 'CRL_REASON_CERTIFICATE_UNHOLD')

    @classmethod
    def find_non_whitelisted_hosts(cls, fqdns):
        yandex_domains = cls.get_yandex_domain_names()
        external_domains = set()
        for fqdn in fqdns:
            fqdn_levels = set(get_domain_levels(fqdn))
            if not bool(fqdn_levels & yandex_domains):
                external_domains.add(fqdn)
        return external_domains

    @classmethod
    def find_non_auto_hosts(cls, fqdns):
        return set()


class RcInternalCA(InternalCA):
    chain_filename = settings.YANDEX_CHAIN

    # Вернуть после отмашки из RUNTIMECLOUD-1150
    # chain_filename = 'rc-internal-ca.pem'


class InternalTestCA(InternalCA):
    chain_filename = 'yandex-test-ca-chain.pem'


class YcInternalCA(InternalCA):
    chain_filename = 'yc-internal-ca.pem'


class CbInternalCA(InternalCA):
    PERMISSION_REQUIRED = 'core.can_issue_cb_internal_ca'
    chain_filename = 'cb-internal-ca.pem'
