import logging
from _ssl import SSLError
from http.client import HTTPException
from urllib.error import URLError

from django.conf import settings
from django.utils.encoding import force_text
from zeep import Client
from zeep.cache import SqliteCache
from zeep.exceptions import TransportError
from zeep.transports import Transport

from intranet.crt.core.ca.exceptions import RetryCaException, CaError

logger = logging.getLogger(__name__)


class ZeepClient(object):
    # Константа NAMESPACE обозначает используемую версию API
    # Можно посмотреть, какие есть версии:
    # wsdl = 'https://test-gcc.globalsign.com/kb/ws/v1/ManagedSSLService?wsdl'
    # username = 'test'
    # password = 'test123'
    #
    # ZeepClient.create(wsdl, username, password).client.namespaces
    #
    # {'ns0': 'https://system.globalsign.com/kb/ws/v1/',
    #  'ns1': 'https://system.globalsign.com/kb/ws/',
    #  'xsd': 'http://www.w3.org/2001/XMLSchema'}
    NAMESPACE = 'ns0'

    @property
    def factory(self):
        return self.client.type_factory(self.NAMESPACE)

    @classmethod
    def create(cls, wsdl_url, username, password):
        try:
            transport = Transport(timeout=settings.ZEEP_TIMEOUT, cache=SqliteCache())
            client = Client(wsdl=wsdl_url, transport=transport)
        except TransportError as error:
            raise RetryCaException from error

        return cls(client, username, password)

    def __init__(self, client, username, password):
        self.client = client
        self.username = username
        self.password = password

    def call(self, method, *args, **kwargs):
        try:
            return getattr(self.client.service, method)(*args, **kwargs)
        except (HTTPException, URLError, SSLError) as e:
            logger.warning('Uncritical exception catched calling %s', method, exc_info=True)
            self._reraise(e, method)
        except Exception as e:
            if self._is_retriable_exception(e):
                logger.warning('Uncritical exception catched calling %s', method, exc_info=True)
                self._reraise(e, method)

            logger.exception('Critical exception catched calling %s', method)
            self._reraise(e, method, retryable=False)

    def _reraise(self, e, method, retryable=True):
        message = '{type} exception catched calling {method}: {message}'.format(
            type='Uncritical' if retryable else 'Critical',
            method=method,
            message=force_text(e),
        )
        exception_type = RetryCaException if retryable else CaError
        raise exception_type(message) from e

    def _is_retriable_exception(self, e):
        message = e.message.lower() if hasattr(e, 'message') else force_text(e).lower()
        retryable = (
            (e.args and ('Bad Gateway' in e.args or isinstance(e.args[0], tuple)))
            or ('unauthorized' in message)
            or ('(500)' in message)
            or ('connection timed out' in message)
        )
        return retryable
