import logging

from django.conf import settings
from django.utils import timezone

from intranet.crt.constants import CERT_STATUS, CERT_TYPE
from intranet.crt.core.controllers.base import BaseController
from intranet.crt.core.models import AssessorCertificate, CertificateType
from intranet.crt.core.notifications import notify
from intranet.crt.utils.random import random_password
from intranet.crt.utils.ssl import create_pfx
from intranet.crt.users.models import CrtUser


PFX_MIME = 'application/x-pkcs12'
DOCX_MIME = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'


class AssessorController(BaseController):
    @staticmethod
    def create_certificate(assessor):
        assessor_type = CertificateType.objects.get(name=CERT_TYPE.ASSESSOR)
        common_name = '{}@assessors.yandex-team.ru'.format(assessor.username)

        certificate = AssessorCertificate.objects.create(
            type=assessor_type,
            ca_name=settings.INTERNAL_CA,
            common_name=common_name,
            email=assessor.email,
            user=assessor,
            requester=CrtUser.objects.robot_crt,
            requested_by_csr=False,
        )

        certificate.controller.issue()

        return certificate

    @staticmethod
    def mark_old_certificates_expiring(assessor, new_cert=None):
        certs = AssessorCertificate.objects.get_all_issued(assessor)
        if new_cert:
            certs = certs.exclude(pk=new_cert.pk)

        expiration_date = timezone.now() + timezone.timedelta(settings.CRT_OLD_CERT_EXPIRATION_DAYS)
        for cert in certs:
            end_date = min(cert.end_date, expiration_date) if cert.end_date else expiration_date
            cert.end_date = end_date
            cert.save(update_fields=['end_date'])

    @classmethod
    def issue_and_notify(cls, assessor):
        certificate = cls.create_certificate(assessor)
        if not certificate or certificate.status != CERT_STATUS.ISSUED:
            logging.warning("Could not issue cert for user %s", assessor.username)
            return

        cls.mark_old_certificates_expiring(assessor, certificate)

        password = random_password()
        cacerts_filename = certificate.controller.ca_cls.get_chain_path(is_ecc=certificate.is_ecc)
        pfx = create_pfx(certificate, password, True, cacerts_filename)

        with open(settings.CRT_ASSESSOR_CRT_MANUAL_FILE, 'rb') as docx_file:
            docx_data = docx_file.read()

        notify(
            certificate.email,
            message_type='assessor_notify',
            subject='Установка клиентских сертификатов',
            password=password,
            attachments=[
                {
                    'filename': '{}.pfx'.format(assessor.username),
                    'content': pfx,
                    'mimetype': PFX_MIME,
                },
                {
                    'filename': 'certificate.docx',
                    'content': docx_data,
                    'mimetype': DOCX_MIME,
                },
            ]
        )
