import abc
import weakref
import collections
import re

from django.db import transaction
from django.db.models import Count
from django.utils import timezone
from django.utils.functional import cached_property

from intranet.crt.constants import ACTION_TYPE, CERT_STATUS
from intranet.crt.core.ca import get_ca_cls, get_ca
from intranet.crt.core.controllers.base import BaseController
from intranet.crt.core.models import Certificate


class BaseCertificateController(BaseController, metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def cert_type(self):
        """ Тип сертификатов, которые обслуживает данный контроллер """
        return None

    @abc.abstractproperty
    def revoke_for_dismissed_user(self):
        """ Показывает нужно ли отзывать сертификаты данного типа для уволенных сотрудников"""
        return False

    @abc.abstractproperty
    def default_internal_ca_template(self):
        return False

    def get_internal_ca_template(self):
        return self.default_internal_ca_template

    def __init__(self, cert):
        self.cert = weakref.proxy(cert)

    @classmethod
    def duplicate_certificates(cls):
        """ Дикт в котором ключ - свежий сертификат, а значение - список дублей """
        return {}

    def process_expiring(self, **kwargs):
        pass

    @property
    def ca_cls(self):
        return get_ca_cls(self.cert.ca_name)

    @cached_property
    def ca(self):
        return get_ca(name=self.cert.ca_name)

    @transaction.atomic
    def issue(self):
        self.ca.issue(self.cert)

    @transaction.atomic
    def revoke(self, user=None, description=None, revoked=None, fake=False):
        if revoked is None:
            revoked = timezone.now()

        self.cert.actions.create(type=ACTION_TYPE.CERT_REVOKE, user=user, description=description)

        if not fake:
            self.ca.revoke(self.cert)

        self.cert.status = CERT_STATUS.REVOKED
        self.cert.revoked = revoked
        self.cert.revoked_by = user
        self.cert.revoke_at = None
        self.cert.save(update_fields=['status', 'revoked', 'revoke_at', 'revoked_by'])

    @transaction.atomic
    def hold(self, user=None, description=None, revoked=None, fake=False):
        if revoked is None:
            revoked = timezone.now()

        self.cert.actions.create(type=ACTION_TYPE.CERT_HOLD, user=user, description=description)

        if not fake:
            self.ca.hold(self.cert)

        self.cert.status = CERT_STATUS.HOLD
        self.cert.revoked = revoked
        self.cert.revoke_at = None
        self.cert.held_by = user
        self.cert.save(update_fields=['status', 'revoked', 'revoke_at', 'held_by'])

    @transaction.atomic
    def unhold(self, user=None, description=None, fake=False):
        self.cert.actions.create(type=ACTION_TYPE.CERT_UNHOLD, user=user, description=description)

        if not fake:
            self.ca.unhold(self.cert)

        self.cert.status = CERT_STATUS.ISSUED
        self.cert.revoked = None
        self.cert.unheld_by = user
        self.cert.save(update_fields=['status', 'revoked', 'unheld_by'])

    @transaction.atomic
    def add_to_hold_queue(self, revoke_at=None, description=None):
        self.cert.actions.create(type=ACTION_TYPE.CERT_ADD_TO_HOLD_QUEUE, description=description)

        if revoke_at is None:
            revoke_at = timezone.now()

        self.cert.revoke_at = revoke_at
        self.cert.save(update_fields=['revoke_at'])

    @transaction.atomic
    def mark_expired(self, description=None):
        self.cert.actions.create(type=ACTION_TYPE.CERT_MARK_EXPIRED, description=description)

        self.cert.status = CERT_STATUS.EXPIRED
        self.cert.revoke_at = None
        self.cert.save(update_fields=['status', 'revoke_at'])

    @transaction.atomic
    def approve_request(self, user, description=None):
        self.cert.actions.create(
            type=ACTION_TYPE.CERT_REQUEST_APPROVED,
            user=user,
            description=description,
        )

        self.cert.status = CERT_STATUS.REQUESTED
        self.cert.save(update_fields=['status'])

    @transaction.atomic
    def reject_request(self, user, description=None):
        self.cert.actions.create(
            type=ACTION_TYPE.CERT_REQUEST_REJECTED,
            user=user,
            description=description,
        )

        self.cert.status = CERT_STATUS.REJECTED
        self.cert.save(update_fields=['status'])

    @transaction.atomic
    def delete_private_key(self, user=None, description=None):
        self.cert.actions.create(
            type=ACTION_TYPE.CERT_PRIVATE_KEY_DELETED,
            user=user,
            description=description,
        )

        self.cert.priv_key_deleted_at = timezone.now()
        self.cert.save(update_fields=['priv_key_deleted_at'])

        self.cert.private_key.delete()


class BaseCsrCertificateController(BaseCertificateController):
    revoke_for_dismissed_user = True

    common_name_re = re.compile(r'^([\w._-]+)@ld\.yandex\.ru$')

    @classmethod
    def duplicate_certificates(cls):
        """ Дубликатами считаются сертификаты выданные на одного пользователя """

        duplicate_user_ids = (
            Certificate.objects
            .values('user_id')
            .annotate(duplicate_count=Count('user_id'))
            .active(cls.cert_type)
            .filter(duplicate_count__gt=1)
            .values_list('user_id', flat=True)
        )

        db_duplicate_certificates = (
            Certificate.objects
            .active(cls.cert_type)
            .filter(user_id__in=duplicate_user_ids)
            .order_by('added')
        )

        duplicate_certificates = collections.defaultdict(list)
        for cert in db_duplicate_certificates:
            duplicate_certificates[cert.user_id].append(cert)

        cert_duplicates = {}
        for certificates in duplicate_certificates.values():
            new_cert = certificates.pop()
            cert_duplicates[new_cert] = certificates

        return cert_duplicates
