import logging
from collections import defaultdict
from itertools import chain

from django.db.models import Q
from django.conf import settings
from django.utils import timezone

from intranet.crt.users.models import CrtUser, CrtGroup
from intranet.crt.constants import CERT_STATUS, CERT_TYPE, CERT_TEMPLATE, ABC_CERTIFICATE_MANAGER_SCOPE, ABC_ADMINISTRATOR_SCOPE
from intranet.crt.core.controllers.certificates import BaseCertificateController
from intranet.crt.core.models import ApproveRequest, Certificate, Host
from intranet.crt.core.notifications import notify
from intranet.crt.utils.startrek import create_st_issue_for_expiring_cert


log = logging.getLogger(__name__)


class HostCertificateController(BaseCertificateController):
    cert_type = CERT_TYPE.HOST
    default_internal_ca_template = None
    revoke_for_dismissed_user = False

    def get_internal_ca_template(self):
        template = CERT_TEMPLATE.WEB_SERVER_ECC if self.cert.is_ecc else CERT_TEMPLATE.WEB_SERVER
        if self.cert.desired_ttl_days is not None and self.cert.desired_ttl_days <= settings.CRT_HOST_CERTS_SHORT_TTL:
            template = CERT_TEMPLATE.WEB_SERVER_ECC_3D if self.cert.is_ecc else CERT_TEMPLATE.WEB_SERVER_3D
        return template

    @staticmethod
    def get_range(days, nearby_days=False):
        """Возвращает кортеж дат начала и конца дня, который будет через days дней"""
        today = timezone.now().replace(hour=0, minute=0, second=0, microsecond=0)
        start_time = today + timezone.timedelta(days=days)
        end_time = today + timezone.timedelta(days=days + 1)
        if nearby_days:
            start_time = today
            end_time = today + timezone.timedelta(days=days)
        interval = (start_time, end_time)
        return interval

    @staticmethod
    def expiring_host_certs_notify(nearby_days=False):
        if nearby_days:
            intervals = [HostCertificateController.get_range(
                settings.CRT_CERT_EXPIRATION_NOTIFICATION_NEARBY_DAYS,
                nearby_days=True,
            )]
        else:
            intervals = [
                HostCertificateController.get_range(days)
                for days in settings.CRT_CERT_EXPIRATION_NOTIFICATION_DAYS
            ]

        q_date_filter = Q()
        for interval in intervals:
            q_date_filter |= Q(end_date__range=interval)

        certs = (
            Certificate.objects
            .filter(
                q_date_filter,
                status=CERT_STATUS.ISSUED,
                type__name=CERT_TYPE.HOST,
            )
            # Не уведомляем о краткосрочных сертификатах
            .exclude(desired_ttl_days__lte=settings.CRT_HOST_CERTS_SHORT_TTL)
            .exclude(notify_on_expiration=False)
            .select_related('requester')
            .prefetch_related('hosts')
            .order_by('id')
        )

        failed_certs = []
        for cert in certs:
            try:
                cert.controller.process_expiring()
            except Exception:
                failed_certs.append(str(cert.pk))
                log.exception('Could not send notification about an expiring certificate')

        if failed_certs:
            log.error(
                'Could not send notifications about the following expiring certificates: %s',
                ','.join(failed_certs),
            )

    def find_expiring_certificate_hosts(self):
        hosts_pks = [host.pk for host in self.cert.hosts.all()]
        candidates_pks = (
            Certificate.objects
            .filter(
                Q(status=CERT_STATUS.ISSUED, end_date__gt=self.cert.end_date) |
                Q(status=CERT_STATUS.NEED_APPROVE),
                type=self.cert.type,
                ca_name=self.cert.ca_name,
                hosts__pk__in=hosts_pks,
            )
            # Не учитываем краткоcрочные сертификаты
            .exclude(desired_ttl_days__lte=settings.CRT_HOST_CERTS_SHORT_TTL)
            .values_list('pk', flat=True)
        )
        linked_host_info = (
            Host.objects
            .filter(
                certificates__pk__in=candidates_pks,
                certificates__hosts__pk__in=hosts_pks,
            )
            .values_list('hostname', 'certificates__pk', 'certificates__status')
        )
        covered_hosts = set()
        certs_to_approve = defaultdict(list)
        for hostname, cert_pk, status in linked_host_info:
            covered_hosts.add(hostname)
            if status == CERT_STATUS.NEED_APPROVE:
                certs_to_approve[cert_pk].append(hostname)
        uncovered_hosts = {host.hostname for host in self.cert.hosts.all()} - covered_hosts
        return uncovered_hosts, certs_to_approve

    def send_expiring_cert_email(self, to_list, expiring_hosts, ticket_to_hosts, expiration_days):
        if not (expiring_hosts or ticket_to_hosts):
            return
        expiring_hosts = sorted(expiring_hosts)
        subject = f'Срок действия сертификата {self.cert.common_name} ({self.cert.ca_name}) подходит к концу'
        kwargs = {}
        if expiring_hosts:
            cr_form_hosts = ','.join(expiring_hosts)
            kwargs['cert_filled_form_url'] = (
                f'{settings.CRT_URL}/certificates/?cr-form=1&cr-form-type=host&'
                f'cr-form-ca_name={self.cert.ca_name}&cr-form-hosts={cr_form_hosts}'
            )
            kwargs['expiring_hosts'] = '\n'.join(expiring_hosts)
        if ticket_to_hosts:
            ticket_to_host_strs = ((k, '\n'.join(sorted(hosts))) for k, hosts in ticket_to_hosts.items())
            kwargs['tickets_to_approve'] = '\n'.join(
                f'В {settings.CRT_STARTREK_URL}{ticket} согласуются хосты:\n{hosts}'
                for ticket, hosts in ticket_to_host_strs
            )
        kwargs['hosts'] = '\n'.join(host.hostname for host in self.cert.hosts.all())
        kwargs['card_url'] = f'{settings.CRT_URL}/certificates/{self.cert.id}?serial_number={self.cert.serial_number}'
        for to in to_list:
            notify(
                to,
                'expiring-host-certs-notify',
                subject=subject,
                cert=self.cert,
                days=expiration_days,
                cc_emails=[settings.CRT_CERT_EXPIRATION_NOTIFICATIONS_CC],
                **kwargs
            )

    def process_expiring(self, **kwargs):
        expiration_days = (self.cert.end_date.date() - timezone.now().date()).days
        expiring_hosts, approve_cert_pk_to_hosts = self.find_expiring_certificate_hosts()
        if not (expiring_hosts or approve_cert_pk_to_hosts):
            return

        emails_to_notify = self.get_emails_to_notify(expiring_hosts, expiration_days)
        ticket_to_hosts = {}
        if expiration_days <= settings.CRT_NOTIFY_ABOUT_NEED_APPROVE_BELOW_DAYS and approve_cert_pk_to_hosts:
            cert_to_ticket = (
                ApproveRequest.objects
                .filter(certificate_id__in=approve_cert_pk_to_hosts)
                .values_list('certificate_id', 'st_issue_key')
            )
            ticket_to_hosts = {
                ticket: approve_cert_pk_to_hosts[pk] for pk, ticket in cert_to_ticket
            }

        self.send_expiring_cert_email(
            emails_to_notify,
            expiring_hosts,
            ticket_to_hosts,
            expiration_days,
        )

        # TODO: слать письма владельцам роботов, когда их можно будет узнать через staff-api
        hosts_for_ticket = expiring_hosts | set(chain.from_iterable(ticket_to_hosts.values()))
        need_create_st_issue = (
            not emails_to_notify
            and hosts_for_ticket
            and self.cert.status == CERT_STATUS.ISSUED
            and self.cert.is_allowed_for_fire_queue
        )
        if need_create_st_issue:
            create_st_issue_for_expiring_cert(self.cert, expiration_days, hosts_for_ticket)

    def get_emails_to_notify(self, expiring_hosts, expiration_days):
        # TODO: предварительно проверить наличие ответственного за сертификат
        # TODO: или руководителя сервиса в ABC

        if self.cert.abc_service_id:
            groups = (
                CrtGroup.objects
                .filter(
                    abc_service_id=self.cert.abc_service_id,
                    role_scope__in={ABC_CERTIFICATE_MANAGER_SCOPE, ABC_ADMINISTRATOR_SCOPE}
                )
            )
            responsibles = list(
                CrtUser.objects
                .filter(staff_groups__in=groups)
                .values_list('email', flat=True)
                .distinct()
            )
            if responsibles:
                return responsibles

        if self.cert.requester.is_active and not self.cert.requester.is_robot:
            return [self.cert.requester.email]
        return []
