import collections

from django.conf import settings
from django.db.models import Count

from intranet.crt.constants import CERT_TYPE, CERT_TEMPLATE, DAYS_IN_MONTH, DAYS_IN_WEEK, DAYS_IN_YEAR
from intranet.crt.core.controllers.certificates import BaseCertificateController
from intranet.crt.core.models import Certificate


TEMPLATES_WITH_TTL = {
    2 * DAYS_IN_WEEK: CERT_TEMPLATE.RC_SERVER_2W,
    DAYS_IN_MONTH: CERT_TEMPLATE.RC_SERVER_1M,
    3 * DAYS_IN_MONTH: CERT_TEMPLATE.RC_SERVER_3M,
    6 * DAYS_IN_MONTH: CERT_TEMPLATE.RC_SERVER_6M,
    DAYS_IN_YEAR: CERT_TEMPLATE.RC_SERVER_1Y,
}


class RcServerCertificateController(BaseCertificateController):
    cert_type = CERT_TYPE.RC_SERVER
    default_internal_ca_template = CERT_TEMPLATE.RC_SERVER
    revoke_for_dismissed_user = False

    def get_internal_ca_template(self):
        if self.cert.desired_ttl_days is None:
            return self.default_internal_ca_template

        return TEMPLATES_WITH_TTL.get(self.cert.desired_ttl_days)

    @classmethod
    def duplicate_certificates(cls):
        """ Дубликатами считаются сертификаты с одинаковым common_name """

        duplicate_common_names = (
            Certificate.objects
            .values('common_name')
            .annotate(duplicate_count=Count('common_name'))
            .active(cls.cert_type, ca_name=settings.RC_INTERNAL_CA)
            .filter(duplicate_count__gt=1)
            .values_list('common_name', flat=True)
        )

        db_duplicate_certificates = (
            Certificate.objects
            .active(cls.cert_type, ca_name=settings.RC_INTERNAL_CA)
            .filter(common_name__in=duplicate_common_names)
            .order_by('added')
        )

        duplicate_certificates = collections.defaultdict(list)
        for cert in db_duplicate_certificates:
            duplicate_certificates[cert.common_name].append(cert)

        cert_duplicates = {}
        for certificates in duplicate_certificates.values():
            new_cert = certificates.pop()
            cert_duplicates[new_cert] = certificates

        return cert_duplicates
