from django import forms
from django.core.exceptions import ValidationError

from intranet.crt.core.models import Certificate
from intranet.crt.users.models import CrtUser


class RequestLinuxCertificateForm(forms.Form):
    device = forms.ChoiceField(label='Устройство')
    password = forms.CharField(label='Пароль', widget=forms.HiddenInput)


class NinjaForm(forms.Form):
    user = forms.ModelChoiceField(
        queryset=CrtUser.objects.active(),
        to_field_name='username',
        required=True,
    )
    is_ios = forms.BooleanField(required=False)

    def __init__(self, user_to_ninja_certs, user_to_quota_limit, requester, **kwargs):
        super(NinjaForm, self).__init__(**kwargs)
        self.users = list(user_to_ninja_certs.keys())
        self.users.remove(requester)
        self.users = [requester] + self.users
        self.user_to_ninja_certs = user_to_ninja_certs
        self.user_to_quota_limit = user_to_quota_limit

        self.form_data = []
        for user in self.users:
            certificates = user_to_ninja_certs[user]
            certificate_fields_list = []
            for cert in certificates:
                field_name = '{}__certificate_{}'.format(user.username, cert.pk)
                cert_field = forms.BooleanField(required=False)
                cert_field.certificate = cert  # Нехорошо, но зато очень удобно
                self.fields[field_name] = cert_field
                certificate_fields_list.append((field_name, cert_field))

            self.form_data.append(
                (
                    user.username,
                    certificates,
                    len(certificates) >= user_to_quota_limit[user],
                    len(certificates),
                    len(certificates) - user_to_quota_limit[user] + 1,
                )
            )

    def clean_user(self):
        user = self.cleaned_data['user']
        if user not in self.users:
            self.add_error('user', f'You cannot request certificate for {user.username}')

        return user

    def clean(self):
        cleaned_data = super(NinjaForm, self).clean()
        if self.errors:
            return self.cleaned_data

        self.certificates_to_revoke = []
        user = self.cleaned_data['user']
        field_prefix = '{}__certificate_'.format(user.username)
        for field_name, value in cleaned_data.items():
            if field_name.startswith(field_prefix) and value:
                self.certificates_to_revoke.append(int(field_name.replace(field_prefix, '')))

        to_remove = len(self.user_to_ninja_certs[user]) - self.user_to_quota_limit[user] + 1
        if len(self.certificates_to_revoke) < to_remove:
            raise ValidationError('Требуется отозвать как минимум {} сертификат(а/ов)'.format(to_remove))

        self.certificates_to_revoke = Certificate.objects.filter(pk__in=self.certificates_to_revoke)
        return self.cleaned_data
