import logging
from datetime import datetime, timedelta

from django.conf import settings

import yenv
from yql.api.v1.client import YqlClient

from intranet.crt.utils.base_command import BaseCommand

log = logging.getLogger(__name__)

YQL_QUERY = """

USE HAHN;
$date_formatter = Python::date_formatter(@@(String?)->Double?@@, @@

def date_formatter(ts):
    try:
        current_datetime = float(ts)
        return int(current_datetime - current_datetime % 5)
    except Exception:
        return 0
@@);

$total_errors = (
    SELECT fielddate,
           COUNT(*) as total_errors,
           COUNT_IF(qloud_instance LIKE 'check-internal-ca-%') as internal_ca_errors
      FROM (
          SELECT $date_formatter(ts) as fielddate, qloud_instance
            FROM [home/logfeller/logs/qloud-router-log/1d/{table_name}]
           WHERE qloud_project = 'tools-access'
             AND qloud_application = 'crt'
             AND qloud_environment = '{qloud_environment}'
             AND qloud_router_name = '{qloud_router_name}'
             AND status LIKE '5__'
       )
  GROUP BY fielddate
);

$errors = (
    SELECT fielddate,
           total_errors,
           IF (internal_ca_errors == 0, total_errors, 0) AS crt_errors
      FROM $total_errors
);

UPSERT INTO {stat_claster}.[{report_path}/daily]
SELECT '{table_name}' as fielddate,
       SUM(crt_errors) as crt_errors,
       SUM(total_errors) as total_errors,
       COUNT(total_errors) * 5 as total_fail_time_sec,
       COUNT_IF(crt_errors != 0) * 5 AS crt_fail_time_sec
FROM $errors;
"""

QLOUD_ROUTER_NAMES = {
    'testing': 'tools-access-crt-test',
    'production': 'tools-crt',
}

STAT_CLASTERS = {
    'testing': 'stat_beta',
    'production': 'stat',
}

REPORT_PATH = 'ExtData/crt/5xx_exclude_internal_ca'


class Command(BaseCommand):
    help = """Считает и загружает в статфейс количество 5xx, исключая ошибки от InternalCA"""

    @staticmethod
    def calculate_crt_errors(report_date=None):
        if report_date is None:
            report_date = datetime.now() - timedelta(days=2)
        # Данные на два дня назад, в YT логи приезжают с задержкой
        table_name = report_date.strftime('%Y-%m-%d')
        qloud_environment = yenv.type

        full_query = YQL_QUERY.format(
            table_name=table_name,
            qloud_environment=qloud_environment,
            qloud_router_name=QLOUD_ROUTER_NAMES[qloud_environment],
            stat_claster=STAT_CLASTERS[qloud_environment],
            report_path=REPORT_PATH,
        )
        yql_client = YqlClient(token=settings.CRT_YQL_OAUTH_TOKEN)
        request = yql_client.query(full_query)
        request.run()
        for table in request.get_results():
            table.fetch_full_data()
        log.info('5xx_exclude_internal_ca for %s sent to Statface' % table_name)

    def _handle(self, *args, **options):
        try:
            self.calculate_crt_errors()
        except Exception:
            log.exception('Can not push 5xx_exclude_internal_ca to Statface')
