import functools
import logging

from django.conf import settings
from django.db import transaction
from ylog.context import log_context

from intranet.crt import constants
from intranet.crt.core.models import HostValidationCode, HOST_VALIDATION_CODE_STATUS
from intranet.crt.core.ca import get_ca
from intranet.crt.core.ca.exceptions import CaError
from intranet.crt.utils.base_command import BaseCommand
from intranet.crt.utils.dns import get_host_txt_codes


log = logging.getLogger(__name__)


class Command(BaseCommand):
    help = "Проверяет, прописаны ли коды верификации для соответствующих доменов " \
           "и отправляет прописанные коды на верификацию."

    @functools.cached_property
    def certum_ca(self):
        return get_ca(settings.CERTUM_CA)

    @functools.cached_property
    def globalsign_ca(self):
        return get_ca(settings.GLOBALSIGN_CA)

    def _handle(self, *args, **options):
        validation_codes = (
            HostValidationCode.objects
            .filter(status=HOST_VALIDATION_CODE_STATUS.validation)
            .select_related('host', 'certificate')
        )
        for validation_code in validation_codes:
            context = {
                'host': validation_code.host.host,
                'certificate': validation_code.certificate.id,
                'validation_code': validation_code.id,
            }
            with log_context(**context):
                try:
                    self.process(validation_code)
                except Exception:
                    log.exception('Error in host %s verification handle', validation_code.host.host)

    @transaction.atomic
    def process(self, validation_code: HostValidationCode):
        host = validation_code.host
        host_name = validation_code.host.host
        name_servers = [name_server.strip() for name_server in host.name_servers.split(',')]

        if validation_code.certificate.ca_name in constants.CA_NAME.CERTUM_CAS:
            ca = self.certum_ca
            globalsign_validation = False
            certum_validation = True
        elif validation_code.certificate.ca_name in constants.CA_NAME.GLOBALSIGN_CAS:
            ca = self.globalsign_ca
            globalsign_validation = True
            certum_validation = False
        else:
            raise RuntimeError(f"Unknown CA for domain validation {repr(validation_code)}")

        codes = get_host_txt_codes(host_name, name_servers, globalsign_validation, certum_validation)
        if validation_code.code not in codes:
            return

        try:
            ca.verify_domain(validation_code)
        except CaError:
            log.exception(
                'Uncaught error while %s verification handle, but maybe domain has already been validated?',
                validation_code.host.host,
            )

        current_status = ca.get_validation_status(validation_code)

        if current_status != validation_code.status:
            validation_code.status = current_status
            validation_code.save()

            if validation_code.status == HOST_VALIDATION_CODE_STATUS.validated:
                host.history.create(
                    action='validated',
                    validation_code=validation_code.code,
                    certificate=validation_code.certificate,
                )

