import datetime

from django.utils import timezone
from django.conf import settings

from intranet.crt.utils.base_command import BaseCommand
from intranet.crt.users.models import CrtUser
from intranet.crt.core.notifications import notify
from intranet.crt.core.models import Certificate
from intranet.crt.constants import CERT_TYPE, CERT_STATUS
from intranet.crt.utils.dns import get_aliases
from intranet.crt.utils.http import CrtSession

BOTIK_MAILING_LIST = 'animals-report@yandex-team.ru'
NOTIFICATION_INTERVAL = datetime.timedelta(days=2)


def get_recipients_emails(host):
    aliases = get_aliases(host)
    recipients = set()

    for host in aliases:
        response = CrtSession().get(
            'https://golem.yandex-team.ru/api/get_host_resp.sbml',
            params={'host': host},
        )
        recipients |= {login.strip() for login in response.content.split(',')}

    return CrtUser.objects.filter(username__in=recipients).values_list('username', flat=True)


class Command(BaseCommand):
    help = """Отправляет уведомления об истекающих сертификатах botik."""

    def _handle(self, *args, **options):
        now = timezone.now()
        threshold = now + datetime.timedelta(30)
        query = (
            Certificate.objects
            .filter(
                status=CERT_STATUS.ISSUED,
                end_date__lte=threshold,
                end_date__gt=now,
                type__name=CERT_TYPE.BOTIK,
            )
            .exclude(expiration_notified_at__gt=now - NOTIFICATION_INTERVAL)
            .select_related('type')
            .prefetch_related('hosts')
        )
        for cert in query:
            notify(
                [BOTIK_MAILING_LIST],
                'expire-soon',
                subject='Срок действия сертификата {} подходит к концу'.format(cert.common_name),
                hosts=', '.join(h.hostname for h in cert.hosts.all()),
                deployed=cert.deployed_at is not None,
                cert=cert,
                cc_emails=[settings.CRT_CERT_EXPIRATION_NOTIFICATIONS_CC],
            )

            cert.expiration_notified_at = now
            cert.save(update_fields=['expiration_notified_at'])
