import datetime
import logging

from django.conf import settings

from intranet.crt.core.models import Certificate
from intranet.crt.constants import CERT_TYPE, CERT_STATUS
from intranet.crt.core.notifications import notify
from intranet.crt.core.utils import get_inums_and_models
from intranet.crt.utils.base_command import BaseCommand

_inums_cache = {}


def get_user_inums(login):
    """Возвращает set всех инвентарников, ассоциированных с сотрудником.
    """
    if login not in _inums_cache:
        data = get_inums_and_models(login, groups=None)
        _inums_cache[login] = {
            item[0] for item in data}

    return _inums_cache[login]


class Command(BaseCommand):
    help = """Отправляет уведомления об истекающих Linux сертификатах.
    Уведомления отправляются за месяц, две недели, неделю и день.

    Тикет: https://st.yandex-team.ru/CERTOR-320
    """

    def _handle(self, *args, **options):
        today = datetime.datetime.utcnow()

        queryset = Certificate.objects.filter(
            type__name=CERT_TYPE.LINUX_PC,
            status=CERT_STATUS.ISSUED
        )

        def process_segment(days_to_expire, date_text):
            expire_at = today + datetime.timedelta(days=days_to_expire)
            expire_range = [
                datetime.datetime.combine(expire_at.date(), datetime.time.min),
                datetime.datetime.combine(expire_at.date(), datetime.time.max),
            ]

            for cert in queryset.filter(end_date__range=expire_range):
                try:
                    inums = get_user_inums(cert.user.username)

                    if cert.pc_inum not in inums:
                        logging.info('User {} does not own device anymore for cert {}'
                                         .format(cert.user.username, cert.id))
                        continue

                    # значит сотрудник всё еще владеет оборудованием
                    # и надо его уведомить о протухающем сертификате
                    recipients = [cert.user.email]
                    subject='Срок действия сертификата {cert.common_name} подходит к концу'.format(
                        cert=cert)

                    notify(
                        recipients,
                        'expire-linuxoids',
                        subject=subject,
                        common_name=cert.common_name,
                        serial=cert.serial_number,
                        expire_date=date_text
                    )

                except Exception:
                    logging.exception('Unable to notify user {} for cert {}'
                                      .format(cert.user.username, cert.id))

        process_segment(30, 'через 30 дней')
        process_segment(14, 'через две недели')
        process_segment(7, 'через одну неделю')
        process_segment(1, 'завтра')
