from datetime import timedelta

from django.conf import settings
from django.db import models
from django.db.models import Prefetch, Q
from django.utils import timezone

from intranet.crt.constants import CERT_TYPE, CERT_STATUS, TAG_SOURCE, ABC_ADMINISTRATOR_SCOPE, ABC_CERTIFICATE_MANAGER_SCOPE
from intranet.crt.core.ca import get_ca


class CertificateQuerySet(models.QuerySet):
    def issued(self):
        return self.filter(status=CERT_STATUS.ISSUED)

    def revoked(self):
        return self.filter(status=CERT_STATUS.REVOKED)

    def active(self, cert_type_name, ca_name=settings.INTERNAL_CA):
        return (
            self
                .issued()
                .filter(
                type__name=cert_type_name,
                ca_name=ca_name,
                revoke_at=None,
            )
        )

    def active_user_hardware_types(self):
        return self.issued().filter(
            ca_name=settings.INTERNAL_CA,
            type__name__in=CERT_TYPE.USER_HARDWARE_TYPES,
        )

    def for_user(self, user):
        if not user.has_perm('core.can_view_any_certificate'):
            from intranet.crt.users.models import CrtGroup
            services = list(
                CrtGroup.objects
                .filter(
                    role_scope__in={ABC_CERTIFICATE_MANAGER_SCOPE, ABC_ADMINISTRATOR_SCOPE},
                    users=user,
                    abc_service__isnull=False,
                ).values_list('abc_service', flat=True)
            )

            return self.filter(Q(abc_service__in=services) | Q(user=user))

        return self

    def prefetch_tags(self, only_active=True):
        from intranet.crt.tags.models import CertificateTag, CertificateTagRelation
        tag_query = CertificateTag.objects.active() if only_active else CertificateTag.objects.all()
        tag_query = tag_query.distinct()  # в разных местах в CRT хочется видеть уникальные теги для сертификата
        tag_rel_query = CertificateTagRelation.objects.select_related('tag')
        if only_active:
            tag_rel_query = tag_rel_query.filter(tag__is_active=True)
        return self.prefetch_related(
            Prefetch('tags', queryset=tag_query),
            Prefetch('tag_relation', queryset=tag_rel_query),
        )

    def for_noc(self):
        taggable_user_types = CERT_TYPE.TAGGABLE_TYPES - CERT_TYPE.NO_USER_TYPES
        taggable_nouser_types = CERT_TYPE.TAGGABLE_TYPES & CERT_TYPE.NO_USER_TYPES

        taggable_user_types_q = Q(type__name__in=taggable_user_types, user__is_active=True)
        taggable_nouser_types_q = Q(type__name__in=taggable_nouser_types)
        return (
            self
                .issued()
                .filter(taggable_user_types_q | taggable_nouser_types_q)
                .order_by('added')
                .prefetch_tags()
                .distinct()
        )

    def for_noc_endpoint(self, since=None):
        query = (
            self
                .issued()
                .order_by('added')
                .select_related('type')
                .prefetch_tags()
        )

        if since is not None:
            query = query.filter(added__gte=since)

        return query

    def filter_serials(self, serial_numbers):
        return self.filter(serial_number__in=serial_numbers)

    def filter_type(self, cert_type):
        return self.filter(type__name=cert_type)

    def imported(self, imported=True):
        return self.filter(is_imported=imported)

    @classmethod
    def _tag_filters(cls, tag):
        return {
            'tag_relation__tag': tag,
            'tag_relation__source': TAG_SOURCE.FILTERS,
        }

    @classmethod
    def _user_type_filters(cls, cert_types, logins):
        return {
            'status': CERT_STATUS.ISSUED,
            'type__in': cert_types,
            'user__username__in': logins,
        }

    def tag_old_sync_query(self, tag, cert_types, logins):
        return (
            self
                .filter(**self._tag_filters(tag))
                .exclude(**self._user_type_filters(cert_types, logins))
        )

    def tag_new_sync_query(self, tag, cert_types, logins):
        return (
            self
                .filter(**self._user_type_filters(cert_types, logins))
                .exclude(**self._tag_filters(tag))
        )

    def without_cert_type_relation(self, tag):
        return (
            self
                .issued()
                .filter(type__in=list(tag.cert_types.all()))
                .exclude(
                tag_relation__tag=tag,
                tag_relation__source=TAG_SOURCE.CERT_TYPE,
            )
        )


class AssessorCertificateQuerySet(models.QuerySet):
    def get_all_issued(self, user=None, include_requested=False):
        if include_requested:
            statuses = [CERT_STATUS.ISSUED, CERT_STATUS.REQUESTED]
        else:
            statuses = [CERT_STATUS.ISSUED]

        res = (
            self
                .filter(
                type__name=CERT_TYPE.ASSESSOR,
                status__in=statuses,
                ca_name=settings.INTERNAL_CA,
            )
        )
        if user is not None:
            res = res.filter(user=user)
        return res

    def can_issue_new(self, user):
        certs = self.get_all_issued(user, include_requested=True)

        if len(certs) > 1:
            return False

        if len(certs) == 1:
            delta = timezone.timedelta(days=settings.CRT_ASSESSOR_EXPIRING_DAYS)
            expire_period = timezone.now() + delta
            if certs[0].status == CERT_STATUS.REQUESTED or certs[0].end_date > expire_period:
                return False

        return True


class ImdmCertificateQuerySet(models.QuerySet):
    pass


class ZombieCertificateQuerySet(models.QuerySet):
    pass


class CertificateTypeQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)

    def filter_names(self, names):
        return self.filter(name__in=names)


class TaskTimestampManager(models.Manager):
    def get_last(self, ts_type):
        return self.get_queryset().filter(type=ts_type, is_success=True).order_by('finish').last()

    def get_last_failed(self, ts_type):
        return self.get_queryset().filter(type=ts_type, is_success=False).last()


class CertificateManager(models.Manager.from_queryset(CertificateQuerySet)):
    def get_expiring_certificates(self, ca_name, days, include_types=None, exclude_types=None):
        """
        Возвращаем сертификаты с end_date меньше текущего числа + days, если они не перезапрошены.
        Считаем сертификат перезапрошенным, если на все его hosts есть сертификаты с лучшей датой такого же типа,
        если сертификат не содержит хостов, то проверяем что на такое common_name есть сертификат с лучшей датой
        такого же типа.

        include_types/exclude_types - должны быть итерируемыми объектами
        Если указан include_types, то exclude_types не учитываются
        """
        from intranet.crt.core.models import Host, CertificateType

        alarm_date = timezone.now() + timedelta(days=days)
        certificates = self.filter(status=CERT_STATUS.ISSUED, ca_name=ca_name)
        if include_types is not None:
            cert_types = set(include_types)
        else:
            ca = get_ca(ca_name)
            cert_types = ca.supported_types
            if exclude_types is not None:
                cert_types -= set(exclude_types)

        expiring_certificates_and_hosts = {}
        cert_types_ids = CertificateType.objects.filter(name__in=cert_types).values_list('id', flat=True)

        for type_id in cert_types_ids:
            expiring_certs = certificates.filter(end_date__lte=alarm_date, type_id=type_id).prefetch_related('hosts')
            not_expiring_certs = certificates.filter(end_date__gt=alarm_date, type_id=type_id)

            expiring_common_names = expiring_certs.values_list('common_name', flat=True)
            covered_common_names = set(not_expiring_certs
                                       .filter(common_name__in=expiring_common_names)
                                       .values_list('common_name', flat=True))

            expiring_hosts = (Host.objects
                              .filter(certificates__in=expiring_certs)
                              .values_list('hostname', flat=True))
            uncovered_hosts = set(Host.objects
                                  .filter(hostname__in=expiring_hosts)
                                  .exclude(certificates__in=not_expiring_certs)
                                  .values_list('hostname', flat=True))

            for cert in expiring_certs:
                if len(cert.hosts.all()) > 0:
                    cert_hosts = []
                    for host in cert.hosts.all():
                        if host.hostname in uncovered_hosts:
                            cert_hosts.append(host.hostname)
                    if cert_hosts:
                        expiring_certificates_and_hosts[cert] = {
                            'reissued': False,
                            'contain_hosts': True,
                            'hosts': sorted(cert_hosts)
                        }
                    else:
                        expiring_certificates_and_hosts[cert] = {'reissued': True}
                else:
                    if cert.common_name not in covered_common_names:
                        expiring_certificates_and_hosts[cert] = {
                            'reissued': False,
                            'contain_hosts': False
                        }
                    else:
                        expiring_certificates_and_hosts[cert] = {'reissued': True}

        return expiring_certificates_and_hosts


class AssessorCertificateManager(models.Manager.from_queryset(AssessorCertificateQuerySet)):
    pass


class ImdmCertificateManager(models.Manager.from_queryset(ImdmCertificateQuerySet)):
    pass


class ZombieCertificateManager(models.Manager.from_queryset(ZombieCertificateQuerySet)):
    pass


class CertificateTypeManager(models.Manager.from_queryset(CertificateTypeQuerySet)):
    pass
