import logging
import re
import constance
import waffle
import yenv
from collections import defaultdict

from django.conf import settings
from django.core import validators
from django.db import models
from django.utils import timezone, encoding
from django.utils.encoding import force_text
from django.utils.functional import cached_property
from django_abc_data.models import AbcService
from library.python.vault_client.errors import ClientError

from intranet.crt.constants import (
    CERT_STATUS,
    CA_NAME,
    ABC_CERTIFICATE_MANAGER_SCOPE,
    CERT_TYPE,
    CERT_EXTENSION,
    HOST_VALIDATION_CODE_STATUS,
)
from intranet.crt.core.fields import EncryptedPrivateKeyField
from intranet.crt.core.ca.registry import get_async_ca_names
from intranet.crt.core.managers import (
    CertificateManager,
    AssessorCertificateManager,
    TaskTimestampManager,
    ImdmCertificateManager,
    ZombieCertificateManager,
    CertificateTypeManager,
)
from intranet.crt.csr import CERT_CSR_CONFIG
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.constance import get_values_set_from_str
from intranet.crt.utils.domain import try_idna_decode
from intranet.crt.utils.dns import get_name_servers, name_servers_are_managed
from intranet.crt.utils.ssl import PemCertificate, PemCertificateRequest
from intranet.crt.utils.tags import suitable_for_field, remove_prefix
from intranet.crt.core.utils import get_yav_client

log = logging.getLogger(__name__)


class CertificateType(models.Model):
    objects = CertificateTypeManager()

    name = models.CharField(max_length=32, unique=True)
    is_active = models.BooleanField(default=True)
    description = models.CharField(max_length=128)

    def __str__(self):
        return self.name


class PrivateKey(models.Model):
    data = EncryptedPrivateKeyField()


class Certificate(models.Model):
    objects = CertificateManager()

    user = models.ForeignKey(CrtUser, null=True, related_name='certificates', on_delete=models.PROTECT)
    requester = models.ForeignKey(CrtUser, related_name='requested_certificates', on_delete=models.PROTECT)
    type = models.ForeignKey(CertificateType, related_name='certificates', on_delete=models.PROTECT)
    status = models.CharField(max_length=20, choices=CERT_STATUS.choices(), default=CERT_STATUS.REQUESTED)
    ca_name = models.CharField(max_length=30, choices=CA_NAME.choices())
    serial_number = models.CharField(max_length=64, null=True, blank=True, db_index=True)
    added = models.DateTimeField(auto_now_add=True, db_index=True)
    begin_date = models.DateTimeField(null=True, blank=True, db_index=True, editable=False)
    end_date = models.DateTimeField(null=True, blank=True, db_index=True, editable=False)
    desired_ttl_days = models.IntegerField(null=True, blank=True)
    private_key = models.OneToOneField(PrivateKey, null=True, on_delete=models.SET_NULL, related_name='certificate')
    is_imported = models.BooleanField(default=False)

    exclude_from_monitoring = models.BooleanField(db_index=True, default=False)

    certificate = models.TextField(null=True, blank=True)
    request = models.TextField(help_text='Для pc, linux-pc и linux-token сертификатов CSR генерятся хелпами.', null=True, blank=True)
    common_name = models.CharField(max_length=255, null=True, blank=True, db_index=True)
    revoke_at = models.DateTimeField(null=True, blank=True)

    held_by = models.ForeignKey(CrtUser, null=True, related_name='held_certificates', on_delete=models.PROTECT)
    unheld_by = models.ForeignKey(CrtUser, null=True, related_name='unheld_certificates', on_delete=models.PROTECT)
    revoked_by = models.ForeignKey(CrtUser, null=True, related_name='revoked_certificates', on_delete=models.PROTECT)

    request_id = models.CharField(max_length=100, null=True, blank=True)
    updated = models.DateTimeField(auto_now=True, db_index=True)
    issued = models.DateTimeField(null=True, blank=True)
    revoked = models.DateTimeField(null=True, blank=True)
    email = models.CharField(max_length=100, null=True, blank=True)
    pc_hostname = models.CharField(max_length=100, null=True, blank=True)
    pc_os = models.CharField(max_length=100, null=True, blank=True)
    pc_serial_number = models.CharField(max_length=100, null=True, blank=True, db_index=True)
    pc_mac = models.CharField(max_length=1000, null=True, blank=True, help_text='Mac адреса, перечисленные через запятую.')
    pc_inum = models.CharField(max_length=100, null=True, blank=True, db_index=True, help_text='Инвентарный номер')
    extended_validation = models.BooleanField(default=False)
    expiration_notified_at = models.DateTimeField(null=True, blank=True, help_text='Время последней отправки письма о истечении сертификата')
    used_template = models.CharField(max_length=100, null=True, blank=True)
    priv_key_deleted_at = models.DateTimeField(null=True, blank=True, help_text='Дата, когда был удален приватный ключ')
    error_message = models.CharField(max_length=8 * 1024, null=True, blank=True)
    device_platform = models.CharField(max_length=100, null=True, blank=True)  # only ninja
    abc_service = models.ForeignKey(
        'django_abc_data.AbcService',
        null=True,
        blank=True,
        default=None,
        on_delete=models.SET_NULL,
    )

    requested_by_csr = models.BooleanField()
    is_reissue = models.BooleanField(default=False)
    uploaded_to_yav = models.NullBooleanField(default=False)
    yav_secret_id = models.CharField(max_length=60, null=True, blank=True, help_text='Id секрета в секретнице')
    yav_secret_version = models.CharField(max_length=60, null=True, blank=True, help_text='Номер верссии секрета')
    is_ecc = models.BooleanField(default=False)
    hardware_request_st_id = models.TextField(null=True, help_text='Ключ тикета на запрос тачки')
    helpdesk_ticket = models.TextField(null=True, help_text="Тикет из HelpDesk")

    notify_on_expiration = models.BooleanField(default=True, db_index=True, help_text='Отправлять ли уведомления при приближении времени истечения сертификата')

    # obsolete
    deployed_at = models.DateTimeField(null=True, blank=True)  # make boolean

    class Meta:
        unique_together = ('ca_name', 'serial_number')

        permissions = (
            ('can_use_console', 'Can use console'),
            ('can_issue_globalsign', 'Can issue globalsign certificates'),
            ('can_issue_cb_internal_ca', 'Can issue certificates on CbInternalCA'),
            ('can_revoke_any_certificate', 'Can revoke any certificate'),
            ('can_issue_courtecy_vpn_certificates', 'Can issue CourtecyVPN certificates'),
            ('can_issue_device_certificates', 'Can issue certificates for macs, pcs and tokens'),
            ('can_issue_ev_certificates', 'Can issue EV certificates'),
            ('can_issue_without_approve', 'Can issue certificate without security approve'),
            ('can_issue_rc_server_certificates', 'Can issue rc-server certificates'),
            ('can_issue_yc_server_certificates', 'Can issue yc-server certificates'),
            ('can_issue_zomb_pc_certificates', 'Can issue zomb-pc certificates'),
            ('can_issue_hypercube_certificates', 'Can issue hypercube certificates'),
            ('can_issue_hypercube_certificates_3d', 'Can issue hypercube certificates 3d'),
            ('can_issue_client_server_certificates', 'Can issue client-server certificates'),
            ('can_issue_bank_client_server_certificates', 'Can issue bank client-server certificates'),
            ('can_issue_sdc_certificates', 'Can issue sdc certificates'),
            ('can_issue_postamate_certificates', 'Can issue postamate certificates'),
            ('can_issue_mdb_certificates', 'Can issue mdb certificates'),
            ('can_issue_tpm_smartcard_1c_certificates', 'Can issue TPM Smartcard 1C Certificates'),
            ('is_responsible_for_any_host', 'Responsible for any host'),
            ('can_have_assessor_certificate', 'Can get an assessor certificate'),
            ('can_issue_imdm_certificates', 'Can issue imdm certificates'),
            ('can_change_abc_service_for_any_certificate', 'Can change ABC service for any certificate'),
            ('can_view_any_certificate', 'Can view any certificate'),
            ('can_download_any_certificate', 'Can download any certificate'),
            ('can_revoke_users_certificates', 'Can revoke users certificates'),
            ('can_edit_certificate_tags', 'Can add and remove tags for any certificate'),
            ('can_issue_certificate_with_tag_oids', 'Can issue certificate with tag OIDs'),
            ('can_issue_temp_pc_certificates', 'Can issue temp-pc certificates'),
            ('can_issue_vpn_1d_certificates', 'Can issue vpn-1d certificates'),
            ('can_issue_vpn_token_certificates_for_self', 'Can issue vpn-token certificates for self'),
            ('can_issue_market_zombie_certificates', 'Can issue zombie certificates for Market'),
            ('can_issue_bank_pc_certificates', 'Can issue bank-pc certificates'),
            (
                'can_issue_device_certificates_for_external',
                '[External] Can issue certificates for macs, pcs and tokens for external',
            ),
        )

    def __repr__(self):
        return (
            '{c.type.name} cert {c.serial_number} from {c.ca_name}, '
            'user: {c.user.username}, status={c.status}'.format(c=self)
        )

    @property
    def priv_key(self):
        return (
            ''
            if self.private_key is None or (
                waffle.switch_is_active('hide_keys') and self.uploaded_to_yav and
                self.requester.username not in constance.config.DELETE_WHITELIST
            )
            else force_text(self.private_key.data)
        )

    @property
    def is_allowed_for_fire_queue(self):
        return (
            (self.ca_name in (CA_NAME.CERTUM_PRODUCTION_CA, CA_NAME.GLOBALSIGN_PRODUCTION_CA) and yenv.type == 'production')
            or
            (self.ca_name in (CA_NAME.CERTUM_TEST_CA, CA_NAME.GLOBALSIGN_TEST_CA) and yenv.type != 'production')
        )

    serial_number_re = re.compile(r'[0-9A-F]+')

    def assert_serial_number(self):
        if self.serial_number is None:
            return

        assert self.serial_number_re.match(self.serial_number) is not None

    def save(self, hosts=None, **kwargs):
        self.assert_serial_number()

        update_fields = kwargs.get('update_fields')
        if update_fields is not None:
            kwargs['update_fields'] = list(update_fields) + ['updated']

        if not self.is_imported:
            if self.user_id is None:
                self.user = self.requester

            if not self.common_name and self.request:
                # при запросе некоторых сертификатов CN не указывается,
                # а вместо этого передается CSR с разными полями.
                # В таком случае, достаем CN из CSR файла
                self.common_name = PemCertificateRequest(self.request).common_name

            if self.id is not None:
                # force_insert устанавливает rest_framework
                # но если сертификат уже был сохранен, то у него
                # уже есть id
                kwargs['force_insert'] = False

            if self.request is None:
                csr_bytes, key_bytes = self.create_csr(hosts)
                self.request = csr_bytes
                self.private_key = PrivateKey.objects.create(data=key_bytes)

        super(Certificate, self).save(**kwargs)

    def wildcards_count(self):
        return self.hosts.filter(hostname__startswith='*.').count()

    def set_certificate(self, certificate):
        """Этот метод должен использоваться вместо простого присвоения
        полю certificate, так как он устанавливает дополнительные свойства."""

        if isinstance(certificate, str):
            certificate = certificate.replace('\r', '')
        if isinstance(certificate, bytes):
            certificate = certificate.replace(b'\r', b'')

        pem_certificate = PemCertificate(certificate)

        self.certificate = certificate
        self.serial_number = pem_certificate.serial_number
        self.end_date = pem_certificate.not_after
        self.begin_date = pem_certificate.not_before

        if self.type.name not in CERT_TYPE.CUSTOM_COMMON_NAME_TYPES:
            self.common_name = try_idna_decode(pem_certificate.common_name)

    def filter_cn_from_sans(self, sans):
        if self.ca_name in CA_NAME.GLOBALSIGN_CAS:
            common_name = force_text(self.common_name.encode('idna'))
            sans_to_filter = {common_name}
            if common_name.startswith(('*.', 'www.')):
                common_name_base = '.'.join(common_name.split('.')[1:])
                sans_to_filter.add(common_name_base)
            sans = [san for san in sans if san not in sans_to_filter]
        return sans

    def get_sans(self):
        """Возвращает SAN как список строк.
        В случае хостового сертификата, там будут домены, в том порядке,
        в котором они были переданы в запросе сертификата."""
        sans = [h.hostname for h in self.hosts.order_by('hostname')]
        return self.filter_cn_from_sans(sans)

    def get_sans2(self, hosts=None):
        """ Необходимо понять как и смерджить это с get_sans"""
        if self.id is None:
            if hosts:
                sans = [force_text(h.hostname.encode('idna')) for h in hosts]
            else:
                return []
        else:
            sans = [force_text(h.hostname.encode('idna')) for h in self.hosts.all()]
        return self.filter_cn_from_sans(sans)

    def custom_extension_support(self):
        return self.ca_name in CA_NAME.CUSTOM_EXTENSION_SUPPORT and self.type.name in CERT_TYPE.TAGGABLE_TYPES

    def build_tags_context(self):
        from intranet.crt.tags.models import CertificateTag
        tags = CertificateTag.objects.actual_for_certificate(self).values_list('name', flat=True)
        context = defaultdict(list)

        for field in CERT_EXTENSION.ALL:
            suitable_tags = [tag for tag in tags if suitable_for_field(tag, field)]
            if suitable_tags:
                context[field] = [remove_prefix(tag) for tag in suitable_tags]
        return context

    def create_csr(self, hosts):
        common_name = self.common_name
        if common_name:
            common_name = force_text(self.common_name.encode('idna'))

        context = {
            'common_name': common_name,
            'email': self.email,
            'sans': self.get_sans2(hosts),
            'country': self.user.country or 'RU',
            'city': self.user.city or 'Moscow',
        }
        csr_config = CERT_CSR_CONFIG[self.type.name](**context)

        if self.custom_extension_support():
            from intranet.crt.tags.controllers import CertificateTagController
            CertificateTagController().update_tags(self)
            tag_context = self.build_tags_context()
            if tag_context:
                csr_config.update_extensions_with_context(tag_context)
        csr_bytes = csr_config.get_csr(is_ecc=self.is_ecc)
        key_bytes = csr_config.get_private_key()

        return csr_bytes, key_bytes

    @models.permalink
    def get_absolute_url(self):
        return 'certificate', [self.pk]

    def add_tag(self, tag, source):
        tag.certificates_rel.objects.create(
            tag=tag,
            certificate=self,
            source=source,
        )

    def remove_tag(self, tag, source):
        tag.certificates_rel.objects.remove_relation(
            tag=tag,
            certificate=self,
            source=source,
        )

    @cached_property
    def controller(self):
        from intranet.crt.core.controllers.certificates import get_certificate_controller
        return get_certificate_controller(self)

    def should_be_shared(self):
        return self.requester.username not in settings.CRT_CERT_REQUESTERS_DO_NOT_SHARE_ACCESS

    def create_new_yav_secret(self, client, data_to_write):
        secret_id = client.create_secret(name='certificate_%s_private_key' % self.serial_number)
        version_id = client.create_secret_version(
            secret_uuid=secret_id,
            value=data_to_write,
        )
        self.yav_secret_id = secret_id
        self.yav_secret_version = version_id
        self.uploaded_to_yav = True

        # TODO при полном переходе на загрузку ключей в секретницу - добавить удаление публичного ключа
        self.save(update_fields=['yav_secret_id', 'yav_secret_version', 'uploaded_to_yav'])
        client.add_user_role_to_secret(secret_id, 'OWNER', login=self.user.username)

        if self.should_be_shared():
            try:
                if self.abc_service:
                    client.add_user_role_to_secret(
                        secret_id,
                        'OWNER',
                        abc_id=self.abc_service.external_id,
                        abc_scope=ABC_CERTIFICATE_MANAGER_SCOPE
                    )

                market_zombies = get_values_set_from_str(constance.config.MARKET_ZOMBIES)
                if self.type.name == CERT_TYPE.ZOMBIE and self.user.username in market_zombies:
                    market_ito_slug = constance.config.MARKET_ABC_SERVICE_SLUG
                    market_ito = AbcService.objects.filter(slug=market_ito_slug).first()
                    if market_ito:
                        client.add_user_role_to_secret(
                            secret_id,
                            'OWNER',
                            abc_id=market_ito.external_id,
                            abc_scope=ABC_CERTIFICATE_MANAGER_SCOPE
                        )
                    else:
                        log.warning(f'ABC Service \'{market_ito_slug}\' not found')

            except ClientError:
                # В проде такой ситуации не должно произойти,
                # но т.к. тестинг секретницы смотрит в прод abc, а тестинг crt - в тестинг abc,
                # то при тестировании такая ситуация возможна
                log.exception(
                    'Can\'t grant access to certificate %s to servicerole in service %s'
                    % (self.serial_number, self.abc_service.slug)
                )

        client.add_user_role_to_secret(secret_id, 'APPENDER', login=settings.CRT_ROBOT)
        client.delete_user_role_from_secret(secret_id, 'OWNER', login=settings.CRT_ROBOT)
        return secret_id, version_id

    def write_private_key_to_yav(self):
        # Записывает приватный ключ и сертификат в секретницу и возвращает id секрета и версии
        if not self.priv_key:
            log.warning('Nothing to write, certificate with id %s has not private key', self.id)
        client = get_yav_client()

        parts = [self.certificate]
        ca_chain_filename = self.controller.ca_cls.get_chain_path(is_ecc=self.is_ecc)
        with open(ca_chain_filename) as chain_file:
            parts.append(chain_file.read())
        parts = [_f for _f in parts if _f]
        pem_cert = '\n'.join(parts)

        data_to_write = [
            {'key': '%s_private_key' % self.serial_number, 'value': self.priv_key},
            {'key': '%s_certificate' % self.serial_number, 'value': pem_cert},
            {'key': '%s_key_cert' % self.serial_number, 'value': self.priv_key + pem_cert},
        ]
        if not self.yav_secret_id:
            return self.create_new_yav_secret(client, data_to_write)
        try:
            # Получим информацию о секрете
            secret = client.get_secret(secret_uuid=self.yav_secret_id, return_raw=False)
            # Получим последнююю версию секрета и создадим  diff от нее,
            # если версий нет, то создадим новую версию
            versions = secret['secret_versions']
            if versions:
                last_version_id = sorted(versions, key=lambda x: x['created_at'], reverse=True)[0]['version']
                new_version_id = client.create_diff_version(
                    last_version_id,
                    diff=data_to_write,
                )
            else:
                new_version_id = client.create_secret_version(
                    secret_uuid=self.yav_secret_id,
                    value=data_to_write,
                )
            self.yav_secret_version = new_version_id
            self.uploaded_to_yav = True

            # TODO при полном переходе на загрузку ключей в секретницу - добавить удаление публичного ключа
            self.save(update_fields=['yav_secret_version', 'uploaded_to_yav'])
            log.info('Private key for certificate %s has uploaded to yav', self.id)
            return self.yav_secret_id, new_version_id
        except ClientError:
            # Если нет доступа к секрету или указа не валидный secret_id, то создадим новый секрет
            log.warning(
                'Secret with id %s does not exist or user %s does not have access',
                self.yav_secret_id,
                settings.CRT_ROBOT,
            )
            return self.create_new_yav_secret(client, data_to_write)

    @property
    def is_async(self):
        return (
            self.ca_name in get_async_ca_names() or
            (
                self.type.name == CERT_TYPE.CLIENT_SERVER and
                waffle.switch_is_active('client_server_should_be_approved')
            ) or
            (
                self.type.name == CERT_TYPE.BANK_CLIENT_SERVER and
                waffle.switch_is_active('bank_client_server_should_be_approved')
            )
        )

    def get_subject_user(self):
        if self.type.name not in CERT_TYPE.USERNAME_IN_COMMON_NAME_TYPES:
            return None
        from intranet.crt.api.v2.certificates.serializers.dispatch import get_serializer
        serializer = get_serializer(self.type.name, self.ca_name)
        return serializer.get_user_from_common_name(self.common_name)

    def get_same_pc_inum_certs(self):
        return Certificate.objects.filter(
            pc_inum=self.pc_inum,
            pc_inum__isnull=False,
        ).exclude(pc_inum='').exclude(pk=self.pk)

    def get_reissue_user(self):
        return self.get_subject_user() or self.user


class AssessorCertificate(Certificate):
    objects = AssessorCertificateManager()

    class Meta:
        proxy = True


class ImdmCertificate(Certificate):
    objects = ImdmCertificateManager()

    class Meta:
        proxy = True


class ZombieCertificate(Certificate):
    objects = ZombieCertificateManager()

    common_name_re = re.compile(r'^(zomb-[\w._-]+)@ld\.yandex\.ru$')
    csr_unit_re = re.compile(r'^zomb$')

    class Meta:
        proxy = True


def create_repr(self, fields):
    return self.__class__.__name__ + ': ' + ', '.join(
        '{0}={1}'.format(key, encoding.smart_str(getattr(self, key)))
        for key in fields)


class Host(models.Model):
    hostname = models.CharField(max_length=255, unique=True)
    certificates = models.ManyToManyField(Certificate, related_name='hosts')

    def __repr__(self):
        return 'Host: {0}'.format(self.hostname.encode('utf-8'))


class HostToApprove(models.Model):
    certificates = models.ManyToManyField(Certificate, related_name='hosts_to_approve', blank=True)
    host = models.CharField(max_length=255, unique=True)
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)
    # список DNS серверов, отвечающих за домен
    # определяется автоматически по хосту
    name_servers = models.CharField(max_length=1024, blank=True, null=True)
    # признак того, что DNS сервера управляются Олегом Гороховым
    # тут будет True если среди name_servers есть либо
    # ns1.yandex.ru, либо ns2.yandex.ru
    managed_dns = models.BooleanField(
        default=False,
        help_text='Сервера под управлением Олега Горохова',
    )
    auto_managed = models.BooleanField(
        default=False,
        help_text='Валидация этого домена автоматизирована',
    )

    # ненулевое значение поля означает, что домен присутствует
    # в списке превалидированных
    globalsign_domain_id = models.CharField(
        max_length=20,
        null=True,
        default=None,
        blank=True,
        validators=[validators.RegexValidator(r"^DSMS\d{11,}$")],
    )

    def __repr__(self):
        return create_repr(self, ('host', 'created', 'updated'))

    def save(self, *args, **kwargs):
        name_servers = get_name_servers(self.host)
        self.name_servers = ','.join(sorted(name_servers))
        self.managed_dns = name_servers_are_managed(name_servers, settings.CRT_MANAGED_NAME_SERVERS)

        super(HostToApprove, self).save(*args, **kwargs)


class HostValidationCode(models.Model):
    code = models.CharField(max_length=128, default=None, blank=True, null=True)
    host = models.ForeignKey(HostToApprove, related_name='validation_codes')
    certificate = models.ForeignKey(Certificate, related_name='validation_codes')
    status = models.CharField(
        max_length=32,
        choices=HOST_VALIDATION_CODE_STATUS,
        default=HOST_VALIDATION_CODE_STATUS.validation,
    )
    created = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now=True)
    expire_at = models.DateTimeField(default=None, blank=True, null=True)

    def __repr__(self):
        return create_repr(self, ('host', 'certificate', 'code'))

    class Meta:
        unique_together = ('host', 'certificate', 'code')

    def is_waiting_for_validation(self):
        now = timezone.now()

        # валидируем только те хосты,
        # зоной которых рулим мы
        if not self.host.managed_dns:
            return False

        # чтобы быть на валидации, код не должен быть протухшим
        if self.expire_at and self.expire_at < now:
            return False

        # а сертификат должен быть в статусе validation
        if self.certificate.status != CERT_STATUS.VALIDATION:
            return False

        return True


class HostToApproveHistory(models.Model):
    host = models.ForeignKey(HostToApprove, related_name='history')
    action = models.CharField(max_length=40)
    created = models.DateTimeField(auto_now_add=True)
    validation_code = models.CharField(max_length=80, null=True, blank=True)
    certificate = models.ForeignKey(Certificate, related_name='host_actions', blank=True, null=True)

    def __repr__(self):
        return create_repr(self, ('host', 'action', 'created'))


class ImapCheckerLog(models.Model):
    num_letters = models.IntegerField()
    num_hosts_to_approve = models.IntegerField()
    # UID последнего обработанного письма
    maximum_uid = models.IntegerField()
    created = models.DateTimeField(auto_now_add=True)

    def __repr__(self):
        return create_repr(self, ('num_letters', 'num_hosts_to_approve', 'maximum_uid', 'created'))


class AliveCheckpoint(models.Model):
    modified_at = models.DateTimeField()


class ApproveRequest(models.Model):
    certificate = models.OneToOneField(Certificate, related_name='approve_request')
    create_date = models.DateTimeField(auto_now_add=True)
    approved = models.BooleanField(default=False)
    update_date = models.DateTimeField(auto_now=True)
    st_issue_key = models.CharField(max_length=20, null=True)
    approver = models.ForeignKey(
        CrtUser, null=True, default=None, related_name='approve_requests', on_delete=models.PROTECT
    )


class TaskTimestamp(models.Model):
    objects = TaskTimestampManager()

    type = models.CharField(max_length=31)
    start = models.DateTimeField(auto_now_add=True)
    finish = models.DateTimeField(null=True, blank=True, db_index=True)
    is_success = models.BooleanField(default=False)
    traceback = models.TextField(null=True, blank=True)


# Модель для проверки возможности читать/писать в базу
class CheckDataBaseAccess(models.Model):
    updated = models.DateTimeField(auto_now=True)
