import re
import collections

from django.conf import settings

from intranet.crt.core.models import Certificate, Host
from intranet.crt.constants import CERT_TYPE, CERT_STATUS
from intranet.crt.users.models import CrtUser

part_re = re.compile(r'([^ ]+ *= *[^ ]+|[\w.-]+)')


def is_username(value):
    return CrtUser.objects.filter(username=value).exists()


def is_certificate_id(value):
    try:
        value = int(value)
    except ValueError:
        return False
    return Certificate.objects.filter(id=value).exists()


def is_certificate_serial(value):
    if isinstance(value, str):  # CERTOR-1723
        value = value.upper()

    return Certificate.objects.filter(serial_number=value).exists()


def is_certificate_type(value):
    return value in CERT_TYPE.active_types()


def is_certificate_status(value):
    return value in CERT_STATUS.all_statuses()


def is_ca_name(value):
    return value in [ca_name for ca_name in settings.AVAILABLE_CA]


checks = [
    (
        is_username, lambda value: {
            'username': [value],
        }
    ),
    (
        is_certificate_id, lambda value: {
            'id': [value],
            'username': ['__any__'],
            'status': ['__any__'],
        }
    ),
    (
        is_certificate_serial, lambda value: {
            'serial_number': [value],
            'username': ['__any__'],
            'status': ['__any__'],
        }
    ),
    (
        is_certificate_type, lambda value: {
            'type': [value],
        }
    ),
    (
        is_certificate_status, lambda value: {
            'status': [value],
        }
    ),
    (
        is_ca_name, lambda value: {
            'ca_name': [value],
        }
    ),
]


def expand(value):
    for check, formatter in checks:
        if check(value):
            return formatter(value)

    return {}


def process(value):
    key, value = value.split('=', 1)
    key = key.strip()
    value = [item.strip() for item in value.split(',')]
    return {key: value}


def parse_search_query(query):
    tokens = collections.defaultdict(list)
    for value in part_re.findall(query):
        if '=' in value:
            new_token = process(value)
        else:
            new_token = expand(value)

        for token, values in new_token.items():
            tokens[token].extend(values)

    return tokens
