import logging

from django.db import DatabaseError
from django.db.models.signals import post_save
from django.dispatch import receiver

from django_idm_api.signals import role_added, role_removed

from intranet.crt.core.controllers.assessor import AssessorController
from intranet.crt.core.models import AssessorCertificate, Certificate
from intranet.crt.tags.controllers import CertificateTagController
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.idm import check_idm_role_group_permissions

log = logging.getLogger(__name__)


@receiver(post_save, sender=Certificate)
def update_certificate_tags(instance=None, **kwargs):
    try:
        CertificateTagController().update_tags(instance)
    except DatabaseError as exc:
        log.exception('Cannot update tags for certificate %s. Error: %s', instance.id, exc)


@receiver(role_added)
def issue_assessor_certificates_on_role_addition(login=None, role=None, **kwargs):
    try:
        role_name = role['role']
    except KeyError:
        return

    if not check_idm_role_group_permissions(role_name, 'can_have_assessor_certificate'):
        return

    try:
        assessor = CrtUser.objects.get(username=login)
    except CrtUser.DoesNotExist:
        log.exception('IDM added new role for user "%s", but CRT does not know him', login)
        return

    if not AssessorCertificate.objects.can_issue_new(assessor):
        return

    AssessorController.issue_and_notify(assessor)


@receiver(role_removed)
def revoke_assessor_certificates_on_role_removal(login=None, role=None, **kwargs):
    try:
        role_name = role['role']
    except KeyError:
        return

    if not check_idm_role_group_permissions(role_name, 'can_have_assessor_certificate'):
        return

    try:
        assessor = CrtUser.objects.get(username=login)
    except CrtUser.DoesNotExist:
        log.exception('IDM removed role for user "%s", but CRT does not know him', login)
        return

    certificates = AssessorCertificate.objects.get_all_issued(assessor)
    for cert in certificates:
        cert.controller.add_to_hold_queue(description='by assessor role removed')
