import envoy
import pipes

from django.conf import settings
from django.utils.encoding import force_text

from intranet.crt.exceptions import CrtBotNotFound
from intranet.crt.utils.http import CrtSession


def shell_exec(shell_command, **arguments):
    """Использовать envoy напрямую небезопасно, особенно когда речь идет
    о командах, в которых используются аргументы полученные от пользователя.
    Эта функция предварительно экранирует все аргументы, подставляет их
    в command, и только после этого запускает envoy.
    """

    def to_string(value):
        return value if isinstance(value, str) else force_text(value)

    escaped = {key: pipes.quote(to_string(value)) for key, value in arguments.items()}

    return envoy.run(shell_command.format(**escaped))


def _get_inum_by_sn_from_bot(serial_number):
    """Возвращает инвентарник устройства по его серийнику из BOT"""
    result = CrtSession().get(
        '{}/api/osinfo.php'.format(settings.CRT_BOT_URL),
        params={'sn': serial_number},
    )
    os = result.json().get('os')
    return os[0]['instance_number'] if os else None


def get_inum_by_sn(serial_number):
    """Возвращает инвентарный номер устройства по его серийному номеру.
    Подробности в задаче https://st.yandex-team.ru/CERTOR-467
    """
    if serial_number is None:
        return None

    inum = _get_inum_by_sn_from_bot('S' + serial_number)
    return inum if inum else _get_inum_by_sn_from_bot(serial_number)


def get_inums_and_models(login, groups=['NOTEBOOKS', 'DESKTOPS']):
    """Возвращает список пар (инвентарный-номер, название-устройства).
    Оба значения в парах — строки."""

    result = CrtSession().set_oauth().get(
        '{}/api/staff/hardware/'.format(settings.CRT_BOT_URL),
        params={'login': login},
    )
    if result.status_code == 404:
        raise CrtBotNotFound('BOT returned 404 with message: {}'.format(result.content))
    if result.status_code != 200:
        raise RuntimeError('Bad status code from BOT: {0}. With message: {1}'.format(
                result.status_code, result.content))

    result = result.json()

    def predicate(item):
        return item['type'] == 'USR' and (not groups or item['group'] in groups)

    return [(item['instance_number'],
             item['sub_group'] + ' ' + item['model'])
            for item in result if predicate(item)]


def old_get_inums_and_models(login, groups=['NOTEBOOKS', 'DESKTOPS']):
    """Старый вариант функции, на ручке без бага OEBS-20332"""

    result = CrtSession().get(
        '{}/api/staff/hw-and-sw-by-login.php'.format(settings.CRT_BOT_URL),
        params={'login': login},
    )
    if result.status_code != 200:
        raise RuntimeError('Bad status code from BOT: {0}. With message: {1}'.format(
                result.status_code, result.content))

    result = result.json()

    def predicate(item):
        return item['type'] == 'USR' and (not groups or item['group'] in groups)

    return [(item['instance_number'],
             item['sub_group'] + ' ' + item['model'])
            for item in result if predicate(item)]


def get_nonauto_hosts(hosts, tlds):
    """Возвращает список хостов, которые не являются
    поддоменами перечисленных tld."""
    response = []
    for host in hosts:
        if not any(host == tld or host.endswith('.' + tld) for tld in tlds):
            response.append(host)
    return response


def expand_wildcard_patterns(fqdn):
    patterns = set()
    # cauth01.dev.yandex-team.ru ->
    # {
    #   'cauth01.dev.yandex-team.ru',
    #   '*.dev.yandex-team.ru',
    #   '*.*.yandex-team.ru'
    # }
    labels = fqdn.rstrip('.').split('.')
    if not all(labels):
        return set()

    for wcs in range(len(labels) - 1):
        patterns.add('.'.join(['*'] * wcs + labels[wcs:]))

    return patterns


def get_yav_client():
    instance = settings.CRT_YAV_INSTANCE
    return instance(authorization=settings.CRT_YAV_TOKEN)


def batched_update(objects, batch_size=1000, **query):
    min_id = objects.earliest('id').id
    max_id = objects.latest('id').id
    for i in range(min_id, max_id + 1, batch_size):
        batch = objects.filter(id__gte=i, id__lt=i + batch_size)
        batch.update(**query)
