import time
import datetime
import logging

from django.shortcuts import render, redirect
from django.http import HttpResponseRedirect, HttpResponse, HttpResponseBadRequest
from django.conf import settings
from django.contrib import messages
from django.urls import reverse
from django.utils import timezone
from django.views.generic import FormView
from django.contrib.auth.decorators import permission_required
from rest_framework import status

from intranet.crt.exceptions import CrtError
from intranet.crt.core.forms import RequestLinuxCertificateForm, NinjaForm
from intranet.crt.core.models import Certificate, PrivateKey, AliveCheckpoint, CertificateType
from intranet.crt.constants import CERT_TYPE, CERT_STATUS
from intranet.crt.core.utils import old_get_inums_and_models
from intranet.crt.utils.iphone import get_iphone_exchange_profile
from intranet.crt.utils.ldap import is_exchange_user
from intranet.crt.utils.mobile import get_os_family
from intranet.crt.utils.random import random_password
from intranet.crt.utils.ssl import create_pfx


def server_error(request):
    template = '500.html'
    response = render(request, template, {'request': request})
    response.status_code = 500
    return response


def index(request):
    return redirect('/request/')


def ping(request):
    modified_dt = AliveCheckpoint.objects.get().modified_at
    unix_time = int(time.mktime(modified_dt.timetuple()))
    return HttpResponse(unix_time, content_type='text/plain')


def migration_text(request):
    template = 'migration.html'
    response = render(request, template)
    return response


@permission_required('core.can_use_console', raise_exception=True)
def console(request):
    template = 'console.html'
    response = render(request, template)
    return response


@permission_required('core.can_use_console', raise_exception=True)
def console_help(request):
    template = 'console-help.html'
    context = {'available_cas': ', '.join(settings.AVAILABLE_CA)}
    response = render(request, template, context)
    return response


class RequestLinuxCertificate(FormView):
    """ Логика такая:
    Рисуем табличку со всеми устройствами сотрудника, далее:
    - если для устройства есть pc сертификат, рисуем кнопку "Заменить сертификат".
    - если для устройства есть linux-pc сертификат, рисуем кнопку "Скачать заново"
      или "Заменить сертификат", если до окончания сущетвующего сертификата осталось
      менее 30 дней.
    - если сертификата нет, то рисуем кнопку "Получить сертификат".

    После нажатия кнопки, рисуем большой видный popup, с паролем.

    Более подробное описание на вики: http://wiki.yandex-team.ru/security/ca/user/dualboot
    """
    form_class = RequestLinuxCertificateForm
    template_name = 'request_linux_certificate.html'

    def get_form(self, form_class=None):
        form = super(RequestLinuxCertificate, self).get_form(form_class)

        inums = old_get_inums_and_models(self.request.user.username)

        form.fields['device'].choices = inums
        form.fields['password'].initial = random_password()
        return form

    @staticmethod
    def _get_used_inums(user):
        certificates = (
            Certificate.objects
            .filter(
                user=user,
                type__name__in=(CERT_TYPE.PC, CERT_TYPE.LINUX_PC),
                status=CERT_STATUS.ISSUED,
            )
            .exclude(pc_inum='')
            .exclude(pc_inum=None)
            .values('pc_inum', 'type__name', 'end_date')
        )
        return {cert['pc_inum']: cert for cert in certificates}

    def get_context_data(self, **kwargs):
        kwargs = super(RequestLinuxCertificate, self).get_context_data(**kwargs)

        used_inums = self._get_used_inums(self.request.user)

        kwargs['data'] = [
            (
                inum,
                model,
                self._need_replace(used_inums, inum),
                self._need_download(used_inums, inum),
                self._need_new(used_inums, inum),
            )
            for inum, model in old_get_inums_and_models(self.request.user.username)
        ]
        kwargs['password'] = random_password()
        kwargs['is_read_only'] = self.request.service_is_readonly

        return kwargs

    @staticmethod
    def _need_replace(used, inum):
        """ CERTOR-363, CERTOR-383
        """
        data = used.get(inum)
        if not data:
            return False

        if data['type__name'] == CERT_TYPE.PC:
            return True

        return data['end_date'] - timezone.now() < datetime.timedelta(90)

    def _need_download(self, used, inum):
        data = used.get(inum)

        private_key_query = PrivateKey.objects.filter(
            certificate__user=self.request.user,
            certificate__type__name=CERT_TYPE.LINUX_PC,
            certificate__pc_inum=inum,
            certificate__status=CERT_STATUS.ISSUED,
        )

        has_cert_with_priv_key = private_key_query.count() > 0
        is_linux_pc = data and data['type__name'] == CERT_TYPE.LINUX_PC
        return is_linux_pc and has_cert_with_priv_key and not self._need_replace(used, inum)

    @staticmethod
    def _need_new(used, inum):
        return inum not in used

    def form_valid(self, form):
        device = form.cleaned_data['device']
        ca_name = settings.INTERNAL_CA

        used_inums = self._get_used_inums(self.request.user)

        # определяем, что следует делать с этим запросом
        need_new = self._need_new(used_inums, device)
        need_replace = self._need_replace(used_inums, device)
        need_download = self._need_download(used_inums, device)
        if need_new and need_download:
            raise CrtError('need_new and need_download should never be True together.')

        rerequest = False
        if need_replace:
            # для замены сертификата сначала холдим все, что
            # были выданы этому сотруднику на этот inum
            # независимо, доменные это были OS или нет
            for cert in Certificate.objects.filter(user=self.request.user,
                                                   pc_inum=device,
                                                   status=CERT_STATUS.ISSUED):
                cert.controller.add_to_hold_queue(
                    description='by api request new linux-pc',
                    revoke_at=timezone.now() + timezone.timedelta(
                        hours=settings.CRT_LINUXPC_HOLD_ON_REISSUE_AFTER_HOURS
                    ),
                )

            # после этого можно выписать новый сертификат
            need_new = True
            rerequest = True

        certificate = None

        if need_download:
            # ищем последний linux-pc сертификат
            pk_query = PrivateKey.objects.filter(
                certificate__user=self.request.user,
                certificate__type__name=CERT_TYPE.LINUX_PC,
                certificate__pc_inum=device,
                certificate__status=CERT_STATUS.ISSUED,
            )

            try:
                last_private_key = pk_query.order_by('-certificate__id')[0]
                certificate = last_private_key.certificate
            except IndexError:
                pass

        if need_new:
            login = self.request.user.username
            common_name = login.lower() + '@ld.yandex.ru'

            linux_pc_type = CertificateType.objects.get(name=CERT_TYPE.LINUX_PC)

            certificate = Certificate.objects.create(
                type=linux_pc_type,
                user=self.request.user,
                requester=self.request.user,
                common_name=common_name,
                email=self.request.user.email,
                status=CERT_STATUS.REQUESTED,
                pc_inum=device,
                ca_name=ca_name,
                is_reissue=rerequest,
                requested_by_csr=False,
            )

            certificate.controller.issue()

        if certificate:
            kwargs = {'pk': certificate.id, 'format': 'pfx'}
            password = form.cleaned_data.get('password', '')
            if password:
                password = '?password=' + password
            url = reverse('api:certificate-download', kwargs=kwargs) + password
            return HttpResponseRedirect(url)
        else:
            raise RuntimeError('Certificate wasn\'t created after the post to /request/ handle.')


class NinjaView(FormView):
    form_class = NinjaForm
    template_name = 'ninja.html'

    def _get_existing_certificates_context(self):
        user_to_ninja_certs = {
            user: []
            for user in [self.request.user] + list(self.request.user.robots.all())
        }
        certificates = (
            Certificate.objects
            .filter(
                status__in=CERT_STATUS.ALIVE_STATUSES,
                type__name__in=(CERT_TYPE.NINJA, CERT_TYPE.NINJA_EXCHANGE),
                user__in=user_to_ninja_certs.keys()
            )
            .select_related('type')
            .order_by('begin_date')
        )
        for cert in certificates:
            user_to_ninja_certs[cert.user].append(cert)

        user_to_quota_limit = {
            user: self._get_user_quota(user) for user in user_to_ninja_certs
        }

        return {
            'user_to_quota_limit': user_to_quota_limit,
            'user_to_ninja_certs': user_to_ninja_certs,
            'requester': self.request.user,
        }

    def _get_user_quota(self, user):
        if user.ninja_certs_quota:
            return user.ninja_certs_quota
        if user.is_robot and user.username.lower().startswith('zomb-'):
            return settings.CRT_USER_NINJA_ZOMB_QUOTA
        return settings.CRT_USER_NINJA_QUOTA

    def get_context_data(self, **kwargs):
        certificates_info = self._get_existing_certificates_context()
        context_data = kwargs.copy()
        if 'form' not in kwargs:
            context_data['form'] = self.get_form(certificates_info=certificates_info)
        context_data.update(certificates_info)

        context_data['certificate_passphrase'] = settings.CRT_NINJA_PFX_PASSWORD
        context_data['is_read_only'] = self.request.service_is_readonly

        return context_data

    def get_form(self, certificates_info, form_class=None):
        if form_class is None:
            form_class = self.get_form_class()

        kwargs = self.get_form_kwargs()
        kwargs.update(certificates_info)
        return form_class(**kwargs)

    @staticmethod
    def new_certificates_exists(requester, user, cert_type):
        issue_delta = datetime.timedelta(minutes=settings.CRT_NINJA_ISSUE_THRESHOLD)
        added_threshold = timezone.now() - issue_delta
        return Certificate.objects.filter(
            type=cert_type,
            user=user,
            requester=requester,
            status__in=(CERT_STATUS.REQUESTED, CERT_STATUS.ISSUED, CERT_STATUS.REVOKED),
            added__gte=added_threshold,
            ca_name=settings.INTERNAL_CA,
        ).exists()

    @staticmethod
    def get_certificate(requester, user, cert_type, device_platform):
        cn_suffix_map = {
            CERT_TYPE.NINJA: 'pda-ld.yandex.ru',
            CERT_TYPE.NINJA_EXCHANGE: 'ld.yandex.ru',
        }
        cn_suffix = cn_suffix_map[cert_type.name]
        common_name = '{}@{}'.format(user.username, cn_suffix)

        certificate = Certificate.objects.create(
            type=cert_type,
            user=user,
            requester=requester,
            common_name=common_name,
            email=user.email,
            status=CERT_STATUS.REQUESTED,
            ca_name=settings.INTERNAL_CA,
            device_platform=device_platform,
            is_reissue=False,
            requested_by_csr=False,
        )

        certificate.controller.issue()

        return certificate

    @staticmethod
    def detect_cert_type(user):
        is_exchange = is_exchange_user(user.username)
        cert_type_name = CERT_TYPE.NINJA_EXCHANGE if is_exchange else CERT_TYPE.NINJA
        return CertificateType.objects.get(name=cert_type_name)

    def detect_device_platform(self, user):
        user_agent = self.request.META.get('HTTP_USER_AGENT', '').lower()
        device_platform = get_os_family(user_agent)
        logging.info(
            'user: {}, user_agent: "{}", device_platform: {}'
            .format(user.username, user_agent, device_platform)
        )
        return device_platform

    def post(self, request, *args, **kwargs):
        """
        Handles POST requests, instantiating a form instance with the passed
        POST variables and then checked for validity.
        """
        form = self.get_form(certificates_info=self._get_existing_certificates_context())
        if form.is_valid():
            return self.form_valid(form)
        else:
            return self.form_invalid(form)

    def form_valid(self, form):
        requester = self.request.user
        user = form.cleaned_data['user']

        if user.username.startswith('robot-'):
            return HttpResponseBadRequest('Requesting certificates for robots is prohibited')

        for certificate in form.certificates_to_revoke:
            certificate.controller.revoke(requester)

        cert_type = self.detect_cert_type(user)

        if self.new_certificates_exists(requester, user, cert_type):
            return HttpResponseBadRequest(
                'Cannot issue mo than 1 {} certificates in {} minutes'
                .format(cert_type.name, settings.CRT_NINJA_ISSUE_THRESHOLD)
            )
        if not (
            # requester должен иметь активный ld-сертификат CERTOR-1876
            requester.has_active_ld_certificates()
            or
            # исключения для некоторых зомби CERTOR-1938
            (
                requester == user
                and
                user.is_pdas_whitelisted()
            )
        ):
            return HttpResponseBadRequest(
                f'User {requester.username} has no active certificates '
                f'to access Yandex network. Issuing prohibited.'
            )

        if form.cleaned_data['is_ios']:
            # kinda hack. Sometimes we can't recognize platform by useragent
            # now it happens on ios, so we override platform here
            device_platform = 'ios'
        elif user == requester:
            device_platform = self.detect_device_platform(requester)
        else:
            device_platform = 'unknown'
        certificate = self.get_certificate(requester, user, cert_type, device_platform)
        cacerts_filename = certificate.controller.ca_cls.get_chain_path(is_ecc=certificate.is_ecc)

        content = create_pfx(
            certificate,
            password=settings.CRT_NINJA_PFX_PASSWORD,
            include_cacerts=True,
            cacerts_filename=cacerts_filename,
        )

        content_type = 'application/x-pkcs12'
        if cert_type.name == CERT_TYPE.NINJA_EXCHANGE and device_platform == 'ios':
            content_type = 'text/plain'
            content = get_iphone_exchange_profile(certificate, content, settings.CRT_NINJA_PFX_PASSWORD)

        response = HttpResponse(content, content_type=content_type, status=status.HTTP_201_CREATED)
        response['Content-Disposition'] = 'attachement; filename="pdas-certificate.p12"'
        return response

    def form_invalid(self, form):
        result = super(NinjaView, self).form_invalid(form)
        if form.errors:
            assert len(form.errors) == 1  # Ожидаем только ошибку о недостаточном количестве отзываемых сертификатов
            common_error_list = list(form.errors.values())[0]
            assert len(common_error_list) == 1  # Ожидаем только ошибку о недостаточном количестве отзываемых сертификатов
            error = common_error_list[0]
            messages.error(self.request, error)
        return result
