import abc

from cryptography import x509

from django.utils.encoding import force_bytes, force_text

from intranet.crt.constants import CERT_EXTENSION


class CsrField(object, metaclass=abc.ABCMeta):
    def __init__(self, required=False, **kwargs):
        self.required = required

    @abc.abstractmethod
    def set_value(self, value):
        pass


class SubjectField(CsrField, metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def oid(self):
        pass

    def __init__(self, value=None, **kwargs):
        super(SubjectField, self).__init__(**kwargs)

        self.value = None
        if value:
            self.set_value(value)

    def set_value(self, value):
        self.value = force_text(value)

    def get_attribute(self):
        return x509.NameAttribute(self.oid, self.value)


class CommonNameSubjectField(SubjectField):
    oid = x509.NameOID.COMMON_NAME


class EmailSubjectField(SubjectField):
    oid = x509.NameOID.EMAIL_ADDRESS


class UnitSubjectField(SubjectField):
    oid = x509.NameOID.ORGANIZATIONAL_UNIT_NAME


class OrganizationSubjectField(SubjectField):
    oid = x509.NameOID.ORGANIZATION_NAME


class OrganizationalUnitSubjectField(SubjectField):
    oid = x509.NameOID.ORGANIZATIONAL_UNIT_NAME


class LocalitySubjectField(SubjectField):
    oid = x509.NameOID.LOCALITY_NAME


class StateSubjectField(SubjectField):
    oid = x509.NameOID.STATE_OR_PROVINCE_NAME


class CountrySubjectField(SubjectField):
    oid = x509.NameOID.COUNTRY_NAME


class DomainComponentSubjectFiled(SubjectField):
    oid = x509.NameOID.DOMAIN_COMPONENT


class ExtensionField(CsrField, metaclass=abc.ABCMeta):
    @abc.abstractmethod
    def get_extension(self, **kwargs):
        pass


class StaticExtensionField(ExtensionField, metaclass=abc.ABCMeta):
    def set_value(self, value):
        raise RuntimeError('Cannot set value for StaticExtensionField')


class BasicConstraintsExtensionField(StaticExtensionField):
    ca = False

    def get_extension(self):
        return x509.BasicConstraints(ca=self.ca, path_length=None)


class ExtendedKeyUsageExtensionField(StaticExtensionField):
    def get_extension(self):
        key_usages = [x509.ExtendedKeyUsageOID.SERVER_AUTH]
        return x509.ExtendedKeyUsage(usages=key_usages)


class KeyUsageExtensionField(ExtensionField):
    def __init__(self, content_commitment=True, **kwargs):
        super(KeyUsageExtensionField, self).__init__(**kwargs)

        self.content_commitment = content_commitment

    def set_value(self, value):
        self.content_commitment = value

    def get_extension(self):
        return x509.KeyUsage(
            digital_signature=True,
            content_commitment=self.content_commitment,
            key_encipherment=True,
            data_encipherment=False,
            key_agreement=False,
            key_cert_sign=False,
            crl_sign=False,
            encipher_only=False,
            decipher_only=False,
        )


class SansExtensionField(ExtensionField):
    def __init__(self, sans=None, **kwargs):
        super(SansExtensionField, self).__init__(**kwargs)

        self.sans = []
        if sans:
            self.set_value(sans)

    def set_value(self, value):
        self.sans = [force_text(san.encode('idna')) for san in value]

    def get_extension(self):
        dns_names = [x509.DNSName(san) for san in self.sans]
        return x509.SubjectAlternativeName(general_names=dns_names)


class CrtCustomExtensionField(ExtensionField):
    oid = None
    extension = None

    def __init__(self, value=None, **kwargs):
        super(CrtCustomExtensionField, self).__init__(**kwargs)
        if value:
            self.value = force_bytes(value)

    def set_value(self, value):
        self.value = str(value)

    def get_extension(self):
        return x509.UnrecognizedExtension(self.oid, self.value)


class CrtWiredTagsExtensionField(CrtCustomExtensionField):
    oid = CERT_EXTENSION.OID[CERT_EXTENSION.WIRED_TAGS]


class CrtWirelessTagsExtensionField(CrtCustomExtensionField):
    oid = CERT_EXTENSION.OID[CERT_EXTENSION.WIRELESS_TAGS]


class CrtVpnTagsExtensionField(CrtCustomExtensionField):
    oid = CERT_EXTENSION.OID[CERT_EXTENSION.VPN_TAGS]


class CrtMobileTagsExtensionField(CrtCustomExtensionField):
    oid = CERT_EXTENSION.OID[CERT_EXTENSION.MOBILE_TAGS]


EXTENSION_FIELD = {
    CERT_EXTENSION.WIRED_TAGS: CrtWiredTagsExtensionField,
    CERT_EXTENSION.WIRELESS_TAGS: CrtWirelessTagsExtensionField,
    CERT_EXTENSION.VPN_TAGS: CrtVpnTagsExtensionField,
    CERT_EXTENSION.MOBILE_TAGS: CrtMobileTagsExtensionField,
}
