import datetime

from django.conf import settings
from django.utils import timezone

from intranet.crt.constants import SECONDS_IN_HOUR
from intranet.crt.core.models import TaskTimestamp
from intranet.crt.monitorings.views.base import BaseMonitoringView
from intranet.crt.tasks.remove_private_keys import RemovePrivateKeysTask
from intranet.crt.tasks.import_internal_ca_certs import ImportInternalCaCertsTask
from intranet.crt.tasks.mark_expired_certs import MarkExpiredCertsTask
from intranet.crt.tasks.revoke.dismissed_users import RevokeDismissedUsersCertsTask
from intranet.crt.tasks.revoke.duplicate_certs import RevokeDuplicateCertsTask
from intranet.crt.tasks.revoke.queued_certs import RevokeQueuedCertsTask
from intranet.crt.tasks.sync_abc_services import SyncAbcServicesTask
from intranet.crt.tasks.sync_assessor_certs import SyncAssessorCertsTask
from intranet.crt.tasks.sync_crl import SyncCrlTask
from intranet.crt.tasks.sync_users import SyncUsersTask
from intranet.crt.tags.tasks.sync_cvs_tags import SyncCvsTagsTask
from intranet.crt.tags.tasks.sync_tags import SyncTagsTask
from intranet.crt.tasks.sync_groups import SyncGroupsTask
from intranet.crt.tasks.save_private_keys_to_yav import SaveKeysToYavTask


class BaseTaskView(BaseMonitoringView):
    """ Monrun мониторинг для проверки свежести выполнения тасков """
    message_template = '{task_type} last finish on {finish} ({hours:.1f} hours). '
    message_not_finished_template = '{task_type} never finished.'

    task_cls = None
    threshold_hours = settings.CRT_DEFAULT_SYNC_MONITOR_HOURS

    def monitoring(self):
        task_type = self.task_cls.task_type
        success_timestamp = TaskTimestamp.objects.get_last(task_type)
        if success_timestamp is None:
            return self.message_not_finished_template.format(task_type=task_type)

        threshold = datetime.timedelta(hours=self.threshold_hours)
        time_since_finish = timezone.now() - success_timestamp.finish

        if time_since_finish > threshold:
            failed_hours = time_since_finish.total_seconds() / SECONDS_IN_HOUR
            error_message = self.message_template.format(
                task_type=task_type,
                finish=success_timestamp.finish.strftime('%d-%m-%Y %H:%M:%S'),
                hours=failed_hours,
            )
            last_timestamp = TaskTimestamp.objects.get_last_failed(task_type)
            if last_timestamp and last_timestamp.traceback:
                error_message += last_timestamp.traceback
            return error_message


class SyncCvsTagsTaskView(BaseTaskView):
    task_cls = SyncCvsTagsTask


class SyncAbcServicesTaskView(BaseTaskView):
    task_cls = SyncAbcServicesTask


class SyncAssessorCertsTaskView(BaseTaskView):
    task_cls = SyncAssessorCertsTask


class SyncUsersTaskView(BaseTaskView):
    task_cls = SyncUsersTask


class SyncTagsTaskView(BaseTaskView):
    task_cls = SyncTagsTask


class SyncCrlTaskView(BaseTaskView):
    task_cls = SyncCrlTask


class ImportInternalCaCertsTaskView(BaseTaskView):
    task_cls = ImportInternalCaCertsTask


class RevokeDismissedUsersCertsTaskView(BaseTaskView):
    task_cls = RevokeDismissedUsersCertsTask


class RevokeDuplicateCertsTaskView(BaseTaskView):
    task_cls = RevokeDuplicateCertsTask


class RevokeQueuedCertsTaskView(BaseTaskView):
    task_cls = RevokeQueuedCertsTask


class MarkExpiredCertsTaskView(BaseTaskView):
    task_cls = MarkExpiredCertsTask


class RemovePrivateKeysTaskView(BaseTaskView):
    task_cls = RemovePrivateKeysTask


class SyncGroupsTaskView(BaseTaskView):
    task_cls = SyncGroupsTask
    threshold_hours = 2


class SavePrivateKeysToYavTaskView(BaseTaskView):
    task_cls = SaveKeysToYavTask
