import re

from django import forms
from django.contrib import admin

from intranet.crt.constants import TAG_FILTER_TYPE
from intranet.crt.tags.models import CertificateTag, TagFilter
from intranet.crt.utils.staff import StaffFilterApi
from intranet.crt.utils.admin import NoDeleteModelAdmin


class CertificateTagFilterCertificateTypeInline(admin.TabularInline):
    verbose_name = 'filter certificate type'
    verbose_name_plural = 'filter certificate types'
    model = CertificateTag.filter_cert_types.through


class CertificateTagStaffFilterInline(admin.TabularInline):
    verbose_name = 'filter'
    verbose_name_plural = 'filters'
    model = CertificateTag.filters.through


class CertificateTagCertificateTypeInline(admin.TabularInline):
    verbose_name = 'certificate type'
    verbose_name_plural = 'certificate types'
    model = CertificateTag.cert_types.through


class CertificateTagAdmin(NoDeleteModelAdmin):
    list_display = ('name', 'type', 'description', 'priority', 'is_active')
    list_editable = ('priority',)
    list_filter = ('name', 'type', 'is_active')
    exclude = ('filter_cert_types', 'filters', 'cert_types', 'certificates')
    inlines = (
        CertificateTagFilterCertificateTypeInline,
        CertificateTagStaffFilterInline,
        CertificateTagCertificateTypeInline,
    )


class TagFilterForm(forms.ModelForm):
    staff_filter_pattern = re.compile(r'^\w{40}$')
    staff_api_pattern = re.compile(r'^\??([\w.]+=[\w,-]+&)*([\w.]+=[\w,-]+)$')

    class Meta:
        model = TagFilter
        fields = ('name', 'type', 'filter', 'description', 'is_broken', 'is_active')

    def clean(self):
        cleaned_data = super(TagFilterForm, self).clean()

        tag_type = cleaned_data['type']
        tag_filter = cleaned_data['filter'].strip()

        if tag_type == TAG_FILTER_TYPE.STAFF_FILTER:
            if self.staff_filter_pattern.match(tag_filter) is None:
                self.add_error('filter', 'Staff filter should be hash')

            api = StaffFilterApi()
            try:
                api.get_filter_id(tag_filter)
            except RuntimeError as error:
                self.add_error('filter', error.message)

        elif tag_type == TAG_FILTER_TYPE.STAFF_API:
            if self.staff_api_pattern.match(tag_filter) is None:
                self.add_error('filter', 'Invalid Staff Api query')

            tag_filter = tag_filter.strip('?&')

        cleaned_data['filter'] = tag_filter

        return cleaned_data


class TagFilterAdmin(NoDeleteModelAdmin):
    form = TagFilterForm

    list_display = ['name', 'type', 'description', 'is_broken', 'is_active']
    list_filter = ['type', 'is_broken', 'is_active']


admin.site.register(CertificateTag, CertificateTagAdmin)
admin.site.register(TagFilter, TagFilterAdmin)
