import logging

from itertools import chain, groupby
from operator import itemgetter
from psycopg2.errors import UniqueViolation

from django.core.exceptions import ObjectDoesNotExist
from django.db.utils import IntegrityError
from django_idm_api.hooks import BaseHooks, AuthHooks

from intranet.crt.constants import TAG_SOURCE
from intranet.crt.users.models import CrtUser
from intranet.crt.tags.models import CertificateTag, TagFilter, CertificateTagRelation


logger = logging.getLogger(__name__)

TAG_ID_TEMPLATE = 'tag_{name}'
TAG_FILTER_ID_TEMPLATE = 'tf_{name}'


class CrtTagsHooks(BaseHooks):
    def info(self):
        all_tagfilters = TagFilter.objects.idm().active().order_by('name').values_list('name', flat=True)
        tagfilters_values = {
            name: {
                'name': name,
                'unique_id': TAG_FILTER_ID_TEMPLATE.format(name=name),
            } for name in all_tagfilters
        }
        all_tags = CertificateTag.objects.active().order_by('name').values_list('name', flat=True)
        tags_values = {
            name: {
                'name': name,
                'unique_id': TAG_ID_TEMPLATE.format(name=name),
            } for name in all_tags
        }

        return {
            'code': 0,
            'roles': {
                'slug': u'type',
                'name': u'Тип',
                'values': {
                    'tagfilter': {
                        'name': 'Тегфильтр',
                        'roles': {
                            'slug': 'tagfilter',
                            'name': 'Тегфильтр',
                            'values': tagfilters_values,
                        }
                    },
                    'tag': {
                        'name': 'Тег по серийному номеру',
                        'roles': {
                            'slug': 'tag',
                            'name': 'Тег',
                            'values': tags_values,
                            'fields': [
                                {
                                    'slug': 'serial_number',
                                    'name': 'Серийный номер',
                                    'required': True,
                                }
                            ]
                        }
                    }
                }
            }
        }

    def _get_filter_by_name(self, name):
        try:
            return TagFilter.objects.idm().active().get(name=name)
        except ObjectDoesNotExist:
            raise Exception(f'Tagfilter with name \'{name}\' does not exists or inactive')

    def _get_tag_by_name(self, name):
        try:
            return CertificateTag.objects.active().get(name=name)
        except ObjectDoesNotExist:
            raise Exception(f'Tag with name \'{name}\' does not exists or inactive')

    def _get_user_by_login(self, login):
        try:
            return CrtUser.objects.get(username=login)
        except ObjectDoesNotExist:
            raise Exception(f'User \'{login}\' does not exists')

    def _get_user_certificate_by_serial_number(self, user, serial_number):
        if not serial_number:
            raise Exception('Serial number is not specified')
        try:
            return user.certificates.get(serial_number__iexact=serial_number)
        except ObjectDoesNotExist:
            raise Exception(f'User\'s \'{user.username}\' certificate with '
                            f'serial number \'{serial_number}\' does not exists')

    def _add_user_to_tagfilter(self, login, role, fields, **kwargs):
        tagfilter = self._get_filter_by_name(role.get('tagfilter'))
        user = self._get_user_by_login(login)

        tagfilter.add_user(user)

    def _remove_user_from_tagfilter(self, login, role, fields, is_fired, **kwargs):
        tagfilter = self._get_filter_by_name(role.get('tagfilter'))
        user = self._get_user_by_login(login)

        tagfilter.remove_user(user)

    def _add_tag_to_certificate(self, login, role, fields, **kwargs):
        tag = self._get_tag_by_name(role.get('tag'))
        user = self._get_user_by_login(login)
        serial_number = fields.get('serial_number')
        certificate = self._get_user_certificate_by_serial_number(user, serial_number)
        try:
            CertificateTagRelation.objects.create(
                tag=tag,
                certificate=certificate,
                source=TAG_SOURCE.IDM,
            )
        except IntegrityError as e:
            # https://wiki.yandex-team.ru/intranet/idm/api/#faq
            # Если роль уже существует, возвращаем ОК
            if isinstance(e.__cause__, UniqueViolation):
                pass
        return {'data': {'serial_number': serial_number}}

    def _remove_tag_from_certificate(self, login, role, fields, is_fired, **kwargs):
        tag = self._get_tag_by_name(role.get('tag'))
        user = self._get_user_by_login(login)
        serial_number = fields.get('serial_number')
        certificate = self._get_user_certificate_by_serial_number(user, serial_number)
        try:
            CertificateTagRelation.objects.remove_relation(
                tag=tag,
                certificate=certificate,
                source=TAG_SOURCE.IDM,
            )
        except ObjectDoesNotExist:
            pass

    def add_role_impl(self, login, role, fields, **kwargs):
        add_role_methods = {
            'tagfilter': self._add_user_to_tagfilter,
            'tag': self._add_tag_to_certificate,
        }
        return add_role_methods[role.get('type')](login, role, fields, **kwargs)

    def remove_role_impl(self, login, role, fields, is_fired, **kwargs):
        remove_role_methods = {
            'tagfilter': self._remove_user_from_tagfilter,
            'tag': self._remove_tag_from_certificate,
        }
        remove_role_methods[role.get('type')](login, role, fields, is_fired, **kwargs)

    def get_all_roles_impl(self):
        tf_roles = TagFilter.objects.active().idm().filter(
            users__isnull=False
        ).values_list('users__username', 'name').order_by('users__username', 'name')
        tf_roles_repr = [(login, {'type': 'tagfilter', 'tagfilter': name}) for login, name in tf_roles]

        tag_roles = CertificateTagRelation.objects.filter(
            source=TAG_SOURCE.IDM
        ).values_list('certificate__user__username', 'certificate__serial_number', 'tag__name').order_by(
            'certificate__user__username', 'certificate__serial_number', 'tag__name',
        )
        tag_roles_repr = [
            (login, [{'type': 'tag', 'tag': name}, {'serial_number': serial_number}])
            for login, serial_number, name in tag_roles
        ]

        all_roles = sorted(chain(tf_roles_repr, tag_roles_repr), key=itemgetter(0))

        for login, user_roles in groupby(all_roles, key=itemgetter(0)):
            yield login, [name for _, name in user_roles]

    def __repr__(self):
        return self.__class__.__name__


def choose(**kwargs):
    if kwargs.pop('tree', None) == 'tags':
        return CrtTagsHooks(**kwargs)
    return AuthHooks(**kwargs)
