import logging

from django.db import models, transaction
from django.db.models import Prefetch, Q

from intranet.crt.constants import TAG_SOURCE, ACTION_TYPE, CERT_TYPE, TAG_FILTER_TYPE

log = logging.getLogger(__name__)


class CertificateTagRelationQuerySet(models.QuerySet):
    @transaction.atomic
    def create(self, tag=None, certificate=None, source=None,  **kwargs):
        super(CertificateTagRelationQuerySet, self).create(
            tag=tag,
            certificate=certificate,
            source=source,
        )

        tag.actions.create(
            type=ACTION_TYPE.CERT_ADD_TAG,
            certificate=certificate,
            description=source,
        )
        log.info('Added tag %s to certificate %s', tag.name, certificate.id)

    def remove_relation(self, tag=None, certificate=None, source=None,  **kwargs):
        relation = self.get(
            tag=tag,
            certificate=certificate,
            source=source,
        )
        relation.delete()

    def certificate_tags_ids(self, certificate, source):
        return (
            self
            .filter(
                source=source,
                certificate=certificate,
            )
            .values_list('tag__pk', flat=True)
            .distinct()
        )

    def broken_cert_type_relation(self, tag):
        return (
            self
            .filter(
                tag=tag,
                source=TAG_SOURCE.CERT_TYPE,
            )
            .exclude(certificate__type__in=list(tag.cert_types.all()))
            .select_related('certificate')
        )


class CertificateTagQuerySet(models.QuerySet):
    def active(self, active=True):
        return self.filter(is_active=active)

    # Вернет также неактивные
    def filter_tags_for_certificate(self, certificate):
        q = Q(
            filters__is_active=True,
            filter_cert_types=certificate.type,
            filters__users=certificate.user,
        )
        # Если у фильтра не проставлен тип сертификата, матчим все тегируемые
        if certificate.type.name in CERT_TYPE.TAGGABLE_TYPES:
            q |= Q(
                filters__is_active=True,
                filter_cert_types__isnull=True,
                filters__users=certificate.user,
            )
        return self.filter(q)

    def new_certificate_filter_tags(self, certificate, exist_tags_ids):
        return self.active().filter_tags_for_certificate(certificate).exclude(pk__in=exist_tags_ids).distinct()

    def new_cert_type_tags(self, certificate):
        return (
            self
            .active()
            .filter(cert_types=certificate.type)
            .exclude(
                cert_relation__source=TAG_SOURCE.CERT_TYPE,
                cert_relation__certificate=certificate,
            )
            .distinct()
        )

    def actual_for_certificate(self, certificate):
        qs = self.filter_tags_for_certificate(certificate) | self.filter(cert_types=certificate.type)
        return qs.active().order_by('priority').distinct()

    def with_cert_types(self):
        return (
            self
            .active()
            .exclude(cert_types__name=None)
            .prefetch_related('cert_types')
        )


class TagFilterQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)

    def not_broken(self):
        return self.filter(is_broken=False)

    def prefetch_tags(self):
        from intranet.crt.tags.models import CertificateTag
        return self.prefetch_related(Prefetch('tags', queryset=CertificateTag.objects.active()))

    def idm(self):
        return self.filter(type=TAG_FILTER_TYPE.IDM_SYSTEM)

    def syncable(self):
        return self.filter(type__in=TAG_FILTER_TYPE.SYNCABLE_TYPES)


class CertificateTagRelationManager(models.Manager.from_queryset(CertificateTagRelationQuerySet)):
    pass


class CertificateTagManager(models.Manager.from_queryset(CertificateTagQuerySet)):
    pass


class TagFilterManager(models.Manager.from_queryset(TagFilterQuerySet)):
    pass
