import re
import logging

from django.core import validators
from django.db import models, transaction

from intranet.crt.constants import TAG_TYPE, TAG_FILTER_TYPE, ACTION_TYPE, TAG_SOURCE
from intranet.crt.core.models import Certificate, CertificateType
from intranet.crt.tags.managers import CertificateTagManager, TagFilterManager, CertificateTagRelationManager
from intranet.crt.users.models import CrtUser

log = logging.getLogger(__name__)


class TagFilter(models.Model):
    objects = TagFilterManager()

    name = models.CharField(max_length=32, unique=True)
    type = models.CharField(max_length=16, choices=TAG_FILTER_TYPE.choices())
    filter = models.TextField(
        help_text='Принимает разные значения, в зависимости от типа. '
                  'API Staff - query string для запрос для staff api. '
                  'Staff фильтр - hash фильтра со staff. '
                  'IDM - поле не используется.',
        null=True, blank=True,
    )
    description = models.CharField(max_length=128)
    is_active = models.BooleanField(default=True)
    is_broken = models.BooleanField(default=False, db_index=True)

    users = models.ManyToManyField(CrtUser, related_name='tag_filters')

    def __str__(self):
        return self.name

    @transaction.atomic
    def add_user(self, user):
        self.users.add(user)
        self.actions.create(
            type=ACTION_TYPE.TAG_FILTER_ADD_USER,
            user=user,
        )
        log.info('Added user %s to filter %s', user.username, self.name)

    @transaction.atomic
    def remove_user(self, user):
        self.users.remove(user)
        self.actions.create(
            type=ACTION_TYPE.TAG_FILTER_REMOVE_USER,
            user=user,
        )
        log.info('Remove user %s from filter %s', user.username, self.name)

    def save(self, *args, **kwargs):
        from intranet.crt.tags.tasks.sync_filters_tags import SyncFilterTagsTask

        # Если произошло снятие is_broken, принудительно синхронизируем фильтр
        if self.pk and not self.is_broken and TagFilter.objects.get(pk=self.pk).is_broken:
            SyncFilterTagsTask.apply_async(kwargs={'filters_pks': [self.pk], 'force': True})
            self.actions.create(
                type=ACTION_TYPE.TAG_FILTER_MARKED_NOT_BROKEN,
                description='\'is_broken\' flag removed',
            )
        # Если фильтр переписали, синхронизируем принудительно
        if self.pk and not (self.filter == TagFilter.objects.get(pk=self.pk).filter):
            super(TagFilter, self).save(*args, **kwargs)
            SyncFilterTagsTask.apply_async(kwargs={'filters_pks': [self.pk], 'force': True})
            self.actions.create(
                type=ACTION_TYPE.TAG_FILTER_CHANGED,
                description='filter hash changed',
            )
            return

        return super(TagFilter, self).save(*args, **kwargs)


class CertificateTagRelation(models.Model):
    class Meta:
        unique_together = ('tag', 'certificate', 'source')

    objects = CertificateTagRelationManager()

    tag = models.ForeignKey('CertificateTag', related_name='cert_relation')
    certificate = models.ForeignKey(Certificate, related_name='tag_relation')
    source = models.CharField(max_length=31, choices=TAG_SOURCE.choices())

    @transaction.atomic
    def delete(self, description=None, **kwargs):
        super(CertificateTagRelation, self).delete(**kwargs)

        if description is None:
            description = self.source

        self.tag.actions.create(
            type=ACTION_TYPE.CERT_REMOVE_TAG,
            certificate=self.certificate,
            description=description,
        )
        log.info('Removed tag %s from certificate %s', self.tag.name, self.certificate.id)


class CertificateTag(models.Model):
    class Meta:
        ordering = ['priority', 'name']

    objects = CertificateTagManager()

    name_validator = validators.RegexValidator(
        re.compile(r'^[\w.-]+$'),
        'Enter a valid "name" consisting of letters, numbers, underscores, dots or hyphens.',
        'invalid'
    )

    name = models.CharField(max_length=32, unique=True, validators=[name_validator])
    type = models.CharField(max_length=16, choices=TAG_TYPE.choices())
    description = models.CharField(max_length=128)
    priority = models.IntegerField(default=0)
    is_active = models.BooleanField(default=True)

    # Если тег связан с фильтром, он будет навешан на все сертификаты пользователей,
    # попадающих под фильтр, и имеющих тип, указанный в filter_cert_types. Если filter_cert_types
    # не указаны, то будет навешан на все тегируемые типы
    filters = models.ManyToManyField(TagFilter, related_name='tags')
    filter_cert_types = models.ManyToManyField(CertificateType, related_name='filter_tags')

    # Если указан cert_types, тег будет навешен на все сертификаты указанного типа,
    # независимо от значений filters и filter_cert_types
    cert_types = models.ManyToManyField(CertificateType, related_name='tags')

    certificates = models.ManyToManyField(
        Certificate,
        through=CertificateTagRelation,
        related_name='tags',
    )

    certificates_rel = CertificateTagRelation

    def __str__(self):
        return self.name
