import collections

from django.utils import timezone
from rest_framework import serializers

from intranet.crt.core.models import Certificate
from intranet.crt.tags.models import CertificateTag, CertificateTagRelation
from intranet.crt.utils import time


class LowerField(serializers.Field):
    def to_representation(self, value):
        return value.lower()


class CertificateTagSerializer(serializers.ModelSerializer):
    fields = ['name']

    class Meta:
        model = CertificateTag

    def to_representation(self, obj):
        return obj.name


class CertificateTagRelationSerializer(serializers.ModelSerializer):
    fields = ['name', 'source']

    class Meta:
        model = CertificateTagRelation

    def to_representation(self, obj):
        return (obj.tag.name, obj.source)


class NocCertListSerializer(serializers.ListSerializer):
    def __init__(self, last_sync=None, **kwargs):
        super(NocCertListSerializer, self).__init__(**kwargs)
        if last_sync is None:
            last_sync = timezone.now()
        self.last_sync = last_sync

    @property
    def certificates(self):
        return super(NocCertListSerializer, self).data

    @property
    def data(self):
        data = collections.OrderedDict()
        data['meta'] = {
            'last_sync': time.aware_to_timestamp(self.last_sync),
        }
        data['certificates'] = self.certificates
        return data


class NocCertSerializer(serializers.ModelSerializer):
    class Meta:
        model = Certificate
        fields = ['serial_number', 'common_name', 'ca_name', 'tags', 'type']
        list_serializer_class = NocCertListSerializer

    serial_number = LowerField()
    common_name = LowerField()
    tags = CertificateTagSerializer(many=True, read_only=True)
    type = serializers.CharField(source='type.name')

    def __init__(self, last_sync=None, **kwargs):
        super(NocCertSerializer, self).__init__(**kwargs)
        if last_sync is None:
            last_sync = timezone.now()
        self.last_sync = last_sync

    @classmethod
    def from_db(cls, last_sync=None):
        db_certificates = Certificate.objects.for_noc()
        return cls(instance=db_certificates, many=True, last_sync=last_sync)
