# coding: utf-8
from __future__ import absolute_import, unicode_literals

import abc
import collections
import constance
import logging

from django.conf import settings
from django.db import transaction

from ids.registry import registry

from intranet.crt.celery_app import app
from intranet.crt.constants import ACTION_TYPE, TASK_TYPE, TAG_FILTER_TYPE, TAG_SOURCE, CERT_TYPE
from intranet.crt.core.models import Certificate, CertificateType
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.tags.models import TagFilter
from intranet.crt.utils.staff import StaffFilterApi
from intranet.crt.utils.startrek import create_issue_for_broken_filter
from intranet.crt.users.models import CrtUser

log = logging.getLogger(__name__)

staff_repository = registry.get_repository(
    'staff', 'person',
    user_agent=settings.CRT_IDS_USER_AGENT,
    oauth_token=settings.CRT_OAUTH_TOKEN,
    timeout=settings.CRT_STAFF_TIMEOUT,
)


class FilterFetcher(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def fetch(self, tag_filter):
        return set()


class StaffApiFilterFetcher(FilterFetcher):
    @staticmethod
    def get_lookup(query):
        return dict([param.split('=') for param in query.split('&')])

    def fetch(self, tag_filter):
        lookup = self.get_lookup(tag_filter.filter)
        lookup['_fields'] = 'login'
        lookup['official.is_dismissed'] = False

        logins = {person['login'] for person in staff_repository.getiter(lookup=lookup)}

        return logins


class StaffFilterFetcher(FilterFetcher):
    def __init__(self):
        self.api = StaffFilterApi()

    def fetch(self, tag_filter):
        return self.api.get_filter_users(tag_filter.filter)


def changes_is_unsafe(tag_filter, current_logins, fetched_logins):
    # Новый фильтр наполняем всегда
    if (
            not tag_filter.pk
            or
            (not current_logins and not tag_filter.actions.filter_content_actions().exists())
    ):
        return tag_filter.actions.exists(), 0

    changes_count = len(fetched_logins ^ current_logins)
    if not changes_count:
        return False, 0.0

    changes_ratio = changes_count * 100.0 / len(current_logins) if current_logins else 100.0

    return (
            # Если изменения единичны, нет смысла считать проценты
            changes_count > int(constance.config.CRT_STAFF_FILTER_CHANGES_THRESHOLD_ABS)
            and
            changes_ratio > int(constance.config.CRT_STAFF_FILTER_CHANGES_THRESHOLD_PERC)
    ), round(changes_ratio, 1)


def aggregate_filter_logins(filters_pks=None, force=False):
    filter_logins = sync_filter_logins(filters_pks=filters_pks, force=force)
    filter_logins.update(collect_idm_filter_logins(filters_pks=filters_pks))
    return filter_logins


def collect_idm_filter_logins(filters_pks=None):
    filter_logins = {}

    qs = TagFilter.objects.active().idm().prefetch_tags()
    if filters_pks:
        qs = qs.filter(pk__in=filters_pks)

    for tag_filter in qs:
        filter_logins[tag_filter] = set(tag_filter.users.values_list('username', flat=True))

    return filter_logins


def sync_filter_logins(filters_pks=None, force=False):
    filter_logins = {}
    fetchers = {
        TAG_FILTER_TYPE.STAFF_FILTER: StaffFilterFetcher(),
        TAG_FILTER_TYPE.STAFF_API: StaffApiFilterFetcher(),
    }

    filters_to_fetch = (
        TagFilter.objects.filter(pk__in=filters_pks).syncable().prefetch_tags()
        if filters_pks
        else
        TagFilter.objects.active().not_broken().syncable().prefetch_tags()
    )

    for tag_filter in filters_to_fetch:
        fetcher = fetchers[tag_filter.type]
        fetched_logins = set(fetcher.fetch(tag_filter))
        log.info('Fetched %s users from %s filter', len(fetched_logins), tag_filter.name)

        current_logins = set(tag_filter.users.values_list('username', flat=True))

        unsafe, changes_ratio = changes_is_unsafe(tag_filter, current_logins, fetched_logins)
        if not force and unsafe:
            tag_filter.is_broken = True
            tag_filter.save(update_fields=['is_broken'])
            log.warning(
                'Filter "{}" changes is too big ({}%). Safe threshold is {}%. Marking up filter as broken.'.format(
                    tag_filter.name, changes_ratio, constance.config.CRT_STAFF_FILTER_CHANGES_THRESHOLD_PERC,
                ))
            issue = create_issue_for_broken_filter(tag_filter, current_logins, fetched_logins)
            tag_filter.actions.create(
                type=ACTION_TYPE.TAG_FILTER_MARKED_BROKEN,
                description='{}: changes exceeded threshold'.format(issue.key),
            )

            continue

        filter_logins[tag_filter] = fetched_logins

    return filter_logins


@transaction.atomic
def update_filter_users(filter_logins):
    for tag_filter, logins in filter_logins.items():
        if tag_filter.type not in TAG_FILTER_TYPE.SYNCABLE_TYPES:
            continue
        db_users = {user.username: user for user in tag_filter.users.all()}

        for login in set(db_users.keys()) - logins:
            tag_filter.remove_user(db_users[login])

        new_logins = logins - set(db_users.keys())
        new_users = CrtUser.objects.filter(username__in=new_logins)
        for user in new_users:
            tag_filter.add_user(user)


@transaction.atomic
def update_certificate_tags(filter_logins):
    tag_logins = collections.defaultdict(set)
    for tag_filter, logins in filter_logins.items():
        for tag in tag_filter.tags.all():
            tag_logins[tag] |= logins
    taggable_types_pks = set(
        CertificateType.objects.filter(name__in=CERT_TYPE.TAGGABLE_TYPES).values_list('pk', flat=True)
    )

    for tag, logins in tag_logins.items():
        # Если к тегу не привязано filter_cert_types, матчим тег на все тегируемые типы
        cert_types = tag.filter_cert_types.all() or CertificateType.objects.filter(pk__in=taggable_types_pks)

        old_certs = Certificate.objects.tag_old_sync_query(tag, cert_types, logins)
        new_certs = Certificate.objects.tag_new_sync_query(tag, cert_types, logins)

        for cert in new_certs:
            cert.add_tag(tag, TAG_SOURCE.FILTERS)

        for cert in old_certs:
            cert.remove_tag(tag, TAG_SOURCE.FILTERS)


class SyncFilterTagsTask(CrtBaseTask):
    task_type = TASK_TYPE.SYNC_FILTER_TAGS
    lock_name = settings.CRT_SYNC_TAGS_LOCK_NAME

    def run(self, timestamp=None, force=False, filters_pks=None):
        filter_logins = aggregate_filter_logins(filters_pks=filters_pks, force=force)
        update_filter_users(filter_logins)
        update_certificate_tags(filter_logins)


SyncFilterTagsTask = app.register_task(SyncFilterTagsTask())
