from django.conf import settings
from django.utils import timezone

from intranet.crt.constants import TASK_TYPE, CERT_TYPE, CERT_STATUS, CERT_TEMPLATE
from intranet.crt.core.controllers.certificates import CERTIFICATE_CONTROLLERS
from intranet.crt.core.models import Certificate, CertificateType
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.ca_export import InternalCaExport
from intranet.crt.utils.crl import InternalCaCrl
from intranet.crt.utils.iterable import slice_iterator


class CertificateMaker(object):
    OID_TEMPLATE_MAP = {
        settings.IMDM_OID: CERT_TEMPLATE.USER_IMDM,
        settings.WIN_PC_AUTO_OID: CERT_TEMPLATE.WIN_PC_AUTO,
        settings.ZOMBIE_OID: CERT_TEMPLATE.ZOMBIE,
        settings.SMARTCARD_OID: CERT_TEMPLATE.SMART_CARD,
        settings.WIN_WH_SHARED_OID: CERT_TEMPLATE.WIN_WH_SHARED,
        settings.WIN_WH_SHARED_OLD_OID: CERT_TEMPLATE.WIN_WH_SHARED,
    }
    TEMPLATE_TYPE_NAME_MAP = {
        CERT_TEMPLATE.USER_IMDM: CERT_TYPE.IMDM,
        CERT_TEMPLATE.WIN_PC_AUTO: CERT_TYPE.WIN_PC_AUTO,
        CERT_TEMPLATE.ZOMBIE: CERT_TYPE.ZOMBIE,
        CERT_TEMPLATE.SMART_CARD: CERT_TYPE.VPN_TOKEN,
        CERT_TEMPLATE.WIN_WH_SHARED: CERT_TYPE.WIN_WH_SHARED,
    }

    def __init__(self, types, users, crl):
        self.types = types
        self.users = users
        self.crl = crl

    @classmethod
    def create(cls):
        types = CertificateType.objects.filter_names(list(cls.TEMPLATE_TYPE_NAME_MAP.values()))
        types = {cert_type.name: cert_type for cert_type in types}
        users = dict(CrtUser.objects.values_list('username', 'id'))
        crl = InternalCaCrl.load()

        return cls(types, users, crl)

    def make(self, cert_info):
        return Certificate(
            is_imported=True,
            requester=CrtUser.objects.robot_crt,
            user_id=self.get_user_id(cert_info),
            type=self.get_type(cert_info),
            status=self.get_status(cert_info),
            ca_name=settings.INTERNAL_CA,
            serial_number=cert_info.serial_number,
            added=timezone.now(),
            end_date=cert_info.not_after,
            common_name=cert_info.common_name,
            revoked=self.get_revoked(cert_info),
            email=cert_info.email,
            used_template=self.OID_TEMPLATE_MAP[cert_info.oid],
            requested_by_csr=False,
            is_reissue=False,
            uploaded_to_yav=False,
            yav_secret_id=False,
            yav_secret_version=False,
        )

    def get_type_name(self, cert_info):
        template = self.OID_TEMPLATE_MAP[cert_info.oid]
        return self.TEMPLATE_TYPE_NAME_MAP[template]

    def get_type(self, cert_info):
        type_name = self.get_type_name(cert_info)
        return self.types[type_name]

    def get_username(self, cert_info):
        type_name = self.get_type_name(cert_info)
        if type_name in CERT_TYPE.NO_USER_TYPES:
            return None

        controller = CERTIFICATE_CONTROLLERS[type_name]
        match = controller.common_name_re.match(cert_info.common_name)
        if match is None:
            return None

        return match.group(1)

    def get_user_id(self, cert_info):
        type_name = self.get_type_name(cert_info)
        if type_name in CERT_TYPE.NO_USER_TYPES:
            return None

        username = self.get_username(cert_info)
        return self.users[username]

    def get_status(self, cert_info):
        if cert_info.not_after < timezone.now():
            return CERT_STATUS.EXPIRED

        if cert_info.revoke_date is None:
            return CERT_STATUS.ISSUED

        if cert_info.serial_number not in self.crl.certs:
            # Если у сетификата есть revoke_date, но его нет в crl, значит когда-то он был в hold
            return CERT_STATUS.ISSUED

        return self.crl.certs[cert_info.serial_number].status

    def get_revoked(self, cert_info):
        cert_status = self.get_status(cert_info)
        return cert_info.revoke_date if cert_status != CERT_STATUS.ISSUED else None

    def is_valid(self, cert_info):
        if cert_info.oid not in self.OID_TEMPLATE_MAP:
            return False

        type_name = self.get_type_name(cert_info)
        if type_name in CERT_TYPE.NO_USER_TYPES:
            return True

        username = self.get_username(cert_info)
        return username in self.users


class ImportInternalCaCertsTask(CrtBaseTask):
    task_type = TASK_TYPE.IMPORT_INTERNAL_CA_CERTS

    @classmethod
    def get_new_serials(cls, serial_numbers):
        new_serial_numbers = set()
        for slice_serials in slice_iterator(serial_numbers, settings.CRT_IN_SLICE_COUNT):
            serials = set(slice_serials)
            db_certs = Certificate.objects.filter_serials(serials)
            db_serial_numbers = {cert.serial_number for cert in db_certs}
            new_serial_numbers |= serials - db_serial_numbers

        return new_serial_numbers

    def run(self, full_sync=False, from_date=None, **kwargs):
        cert_maker = CertificateMaker.create()

        export = InternalCaExport.load(full_sync, from_date)
        certs_info = list(export.certs_info.values())
        serial_numbers = [info.serial_number for info in certs_info if cert_maker.is_valid(info)]

        new_serial_numbers = self.get_new_serials(serial_numbers)

        new_certificates = []
        for serial_number in new_serial_numbers:
            cert_info = export.certs_info[serial_number]
            certificate = cert_maker.make(cert_info)
            new_certificates.append(certificate)

        if full_sync:
            Certificate.objects.bulk_create(new_certificates)
        else:
            for certificate in new_certificates:
                certificate.save()
