import constance
import logging

from django.utils import timezone
from django.db.models import Q

from intranet.crt.core.models import Certificate
from intranet.crt.core.notifications import notify
from intranet.crt.constants import (
    TASK_TYPE,
    CERT_TYPE,
    ABC_CERTIFICATE_MANAGER_SCOPE,
    CERT_STATUS,
    ABC_ADMINISTRATOR_SCOPE
)
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.users.models import CrtGroup, CrtUser
from intranet.crt.utils.constance import get_values_set_from_str

log = logging.getLogger(__name__)


def send_notify(requester, secret_id, version_id, certificate, abc_cert_users=None):
    notify(
        requester.email,
        'certificate-was-issued-and-private-key-has-uploaded-to-yav',
        subject='Сертификат готов',
        certificate=certificate,
        hosts=[host.hostname for host in certificate.hosts.all()],
        secret_id=secret_id,
        version_id=version_id,
    )
    if abc_cert_users:
        emails = [user.email for user in abc_cert_users]
        notify(
            emails,
            'certificate-for-abc-service-was-issued-and-private-key-has-uploaded-to-yav',
            subject='Сертификат готов',
            certificate=certificate,
            hosts=[host.hostname for host in certificate.hosts.all()],
            secret_id=secret_id,
            version_id=version_id,
            requester=requester.username,
            abc_service_name=certificate.abc_service.name,
        )


def get_users_to_notify(certificate):
    abc_groups = CrtGroup.objects.filter(
        abc_service_id=certificate.abc_service_id,
        role_scope__in=(ABC_CERTIFICATE_MANAGER_SCOPE, ABC_ADMINISTRATOR_SCOPE),
    )
    abc_users = CrtUser.objects.filter(staff_groups__in=abc_groups).exclude(id=certificate.user_id)
    return abc_users


class SaveKeysToYavTask(CrtBaseTask):
    task_type = TASK_TYPE.SAVE_KEYS_TO_YAV

    def run(self, **kwargs):
        now = timezone.now()

        market_zombies = get_values_set_from_str(constance.config.MARKET_ZOMBIES)
        market_zombies_ids = set(CrtUser.objects.filter(username__in=market_zombies).values_list('pk', flat=True))
        certificates = (
            Certificate.objects
            .filter(
                begin_date__lt=now,
                requested_by_csr=False,
                uploaded_to_yav=False,
                status=CERT_STATUS.ISSUED,
            )
            .filter(
                Q(type__name=CERT_TYPE.HOST)
                |
                Q(type__name=CERT_TYPE.ZOMBIE, user_id__in=market_zombies_ids)
            )
            .exclude(
                private_key=None,
                begin_date=None,
            )
            .select_related('user')
            .order_by('pk')
        )
        # При миграции всем старым сертификатам begin_date проставим None, отбросим их в фильтре
        # order_by нужен для прохождения юнит-тестов на sqlite и postgresql

        for certificate in certificates:
            secret_id, version_id = certificate.write_private_key_to_yav()
            if certificate.type.name == CERT_TYPE.HOST and certificate.should_be_shared():
                abc_scope_users = get_users_to_notify(certificate)
                send_notify(certificate.user, secret_id, version_id, certificate, abc_cert_users=abc_scope_users)
