import logging
import itertools

from django.conf import settings
from django.contrib.auth.models import Permission
from django.db.models import Count, Max, Q
from django.utils import timezone

from intranet.crt.constants import TASK_TYPE, CERT_TYPE, CERT_STATUS
from intranet.crt.core.controllers.assessor import AssessorController
from intranet.crt.core.models import AssessorCertificate
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.users.models import CrtUser

log = logging.getLogger(__name__)


class SyncAssessorCertsTask(CrtBaseTask):
    task_type = TASK_TYPE.SYNC_ASSESSOR_CERTS

    def run(self, **kwargs):
        perm = Permission.objects.get(codename='can_have_assessor_certificate')
        expire_period = timezone.now() + timezone.timedelta(settings.CRT_ASSESSOR_EXPIRING_DAYS)

        assessors = CrtUser.objects.with_permission(perm)
        assessors_with_certs = (
            assessors
            .filter(
                certificates__type__name=CERT_TYPE.ASSESSOR,
                certificates__status__in=[CERT_STATUS.ISSUED, CERT_STATUS.REQUESTED],
                certificates__ca_name=settings.INTERNAL_CA,
            )
        )
        assessors_to_renew = (
            assessors_with_certs
            .filter(
                certificates__status=CERT_STATUS.ISSUED,
            )
            .annotate(
                certs_num=Count('certificates'),
                end_date=Max('certificates__end_date'),
            )
            .filter(
                Q(certs_num=0) | (Q(certs_num=1) & Q(end_date__lte=expire_period))
            )
        )
        assessors_without_certs = assessors.exclude(pk__in=assessors_with_certs)

        assessors_ids = {assessor.pk for assessor in assessors}
        certs_to_revoke = (
            AssessorCertificate.objects
            .filter(
                type__name=CERT_TYPE.ASSESSOR,
                status=CERT_STATUS.ISSUED,
                ca_name=settings.INTERNAL_CA,
            )
            .exclude(
                user__pk__in=assessors_ids,
            )
        )

        for assessor in itertools.chain(assessors_without_certs, assessors_to_renew):
            AssessorController.issue_and_notify(assessor)

        for cert in certs_to_revoke:
            cert.controller.add_to_hold_queue(description='by assessor roles sync task')
