import logging

from django.conf import settings
from django.db.models import Q
from django.utils import timezone

from intranet.crt.constants import CERT_STATUS, TASK_TYPE
from intranet.crt.core.models import Certificate
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.utils.crl import InternalCaCrl

log = logging.getLogger(__name__)


class SyncCrlTask(CrtBaseTask):
    task_type = TASK_TYPE.SYNC_CRL

    @classmethod
    def mark_revoked(cls, crl):
        to_hold_q = Q(status=CERT_STATUS.ISSUED, serial_number__in=crl.hold_certs)
        to_revoke_q = Q(
            status__in=[CERT_STATUS.ISSUED, CERT_STATUS.HOLD],
            serial_number__in=crl.revoked_certs,
        )
        db_certificates = (
            Certificate.objects
            .filter(ca_name=settings.INTERNAL_CA)
            .filter(to_hold_q | to_revoke_q)
        )

        for cert in db_certificates:
            crl_cert = crl.certs[cert.serial_number]

            log.warning('Certificate {} {} beyond crt'.format(cert.serial_number, crl_cert.status))

            revoked = crl_cert.revoke_date
            if crl_cert.status == CERT_STATUS.HOLD:
                cert.controller.hold(description='by crl', revoked=revoked, fake=True)
            elif crl_cert.status == CERT_STATUS.REVOKED:
                cert.controller.revoke(description='by crl', revoked=revoked, fake=True)

    @classmethod
    def hold_to_issued(cls, crl):
        db_certificates = (
            Certificate.objects
            .filter(
                ca_name=settings.INTERNAL_CA,
                status=CERT_STATUS.HOLD,
                end_date__gt=timezone.now(),  # Не берем истекшие серты, которые пока не expired
                revoked__lt=crl.last_update,  # Отозваны до публикации CRL
            )
            .exclude(serial_number__in=crl.certs.keys())
        )

        for cert in db_certificates:
            log.warning('Certificate {} unhold beyond crt'.format(cert.serial_number))

            cert.controller.unhold(description='by crl', fake=True)

    def run(self, **kwargs):
        crl = InternalCaCrl.load()

        self.mark_revoked(crl)
        self.hold_to_issued(crl)
