import logging
import yenv

from django.conf import settings
from django.db import transaction

from ids.registry import registry
from django_abc_data.models import AbcService

from intranet.crt.constants import (
    TASK_TYPE,
    CRT_GROUP_TYPE,
    INCONSISTENCY_TYPE,
    STAFF_SYNC_LIMIT,
    REMOVE_USERS_FROM_SCOPE_THRESHOLD,
    USE_THRESHOLD_IF_USERS_GREATER,
    SKIP_INCONSISTENT_GROUPS_THRESHOLD,
)
from intranet.crt.monitorings.models import INCONSISTENCY_TYPE, add_inconsistencies
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.users.models import CrtUser, CrtGroup
from intranet.crt.exceptions import CrtSyncException

BATCH_SIZE = 256
ATTR_TO_SYNC = ['name', 'type', 'is_deleted', 'role_scope', 'abc_service_id', 'url']

log = logging.getLogger(__name__)

yenv.type = getattr(settings, 'CRT_IDS_OVERRIDE_ENV_TYPE', yenv.type)
staff_repo = registry.get_repository(
    'staff', 'groupmembership',
    user_agent=settings.CRT_IDS_USER_AGENT,
    oauth_token=settings.CRT_OAUTH_TOKEN,
    timeout=settings.CRT_STAFF_TIMEOUT,
)


def get_service_roles(skip_threshold):
    abc_services = {service.external_id: service.id for service in AbcService.objects.all()}
    log.info('Get a list of serviceroles from the system')
    lookup = {
        'group.type': CRT_GROUP_TYPE.SERVICE_ROLE,  # Выбираем только servicerole
        'group.is_deleted': False,
        '_limit': STAFF_SYNC_LIMIT,
        '_fields': ','.join((
            'group.id',
            'group.role_scope',
            'group.is_deleted',
            'group.name',
            'group.url',
            'person.login',
            'group.parent.service.id',
        )),
    }

    result = {}
    inconsistent_groups_descriptions = []
    total_count = 0
    for membership in staff_repo.getiter(lookup=lookup):
        total_count += 1
        group = membership['group']

        missing_fields = ','.join(
            field
            for field in ['id', 'name', 'is_deleted', 'is_deleted', 'role_scope', 'parent', 'url']
            if field not in group
        )
        if missing_fields:
            inconsistent_groups_descriptions.append(
                f'Group #{group.get("id")} has no fields: {missing_fields}'
            )
            continue

        if group['id'] not in result:
            result[group['id']] = {
                'external_id': group['id'],
                'type': CRT_GROUP_TYPE.SERVICE_ROLE,
                'name': group['name'],
                'is_deleted': group['is_deleted'],
                'url': group['url'],
                'role_scope': group['role_scope'],
                'abc_service_id': abc_services.get(group['parent']['service']['id']),
                'users': set()
            }
        result[group['id']]['users'].add(membership['person']['login'])

    skipped_count = len(inconsistent_groups_descriptions)
    if total_count == 0 or skipped_count / total_count < skip_threshold:
        add_inconsistencies(inconsistent_groups_descriptions, INCONSISTENCY_TYPE.GROUP)
    else:
        raise CrtSyncException(f'Too many inconsistent groups: {inconsistent_groups_descriptions}')
    log.info('Receive %s remote serviceroles', len(result))
    return list(result.values())


@transaction.atomic
def update_users(group, local_users, remote_group_users, threshold):
    log.info('Start synchronization users in group %s', group.name)
    group_users = {user.username for user in group.users.all()}
    # Удаляем пользователей, которых нам не передал staff, если количество таких пользователей
    # не больше threshold
    current_group_users_count = len(group_users)
    if (
        current_group_users_count > USE_THRESHOLD_IF_USERS_GREATER
        and len(group_users - remote_group_users)/current_group_users_count > threshold
    ):
        log.error(
            'Do not remove users from group %s, deleted users - %s%%, threshold - %s%%',
            group.name,
            len(group_users - remote_group_users)/current_group_users_count*100,
            threshold*100,
        )
        raise CrtSyncException()

    log.info('Remove users from group %s', group.name)
    for user in group_users - remote_group_users:
        group.users.remove(local_users.get(user))
    log.info('Users has removed from group %s', group.name)

    # Проверим, что у нас уже есть такой пользователь и добавим его в группу,
    # иначе добавим его при следующей синхронизации
    for user in remote_group_users - group_users:
        log.info('Add users to group %s', group.name)
        _user = local_users.get(user)
        if _user is not None:
            group.users.add(_user)
        log.info('Users has added to group %s', group.name)
    log.info('Users synchronization from group %s is complete', group.name)


def update_groups(remote_groups, threshold):
    log.info('Starting serviceroles synchronization with staff')
    local_groups = {
        group.external_id: group
        for group in CrtGroup.objects.all().prefetch_related('users')
    }
    local_users = {user.username: user for user in CrtUser.objects.all()}
    group_to_save = []
    group_to_save_batches = []
    group_to_save_count = 0

    for remote_group in remote_groups:
        need_save = False
        if remote_group['external_id'] in local_groups:
            group = local_groups.pop(remote_group['external_id'])

            for attr in ATTR_TO_SYNC:
                local_value = getattr(group, attr)
                remote_value = remote_group.get(attr)
                if local_value != remote_value:
                    setattr(group, attr, remote_value)
                    need_save = True
        else:
            # Для создания many_to_many нужен pk, приходится создавать объект
            group = CrtGroup.objects.create(
                url=remote_group['url'],
                role_scope=remote_group['role_scope'],
                type=remote_group['type'],
                external_id=remote_group['external_id'],
                name=remote_group['name'],
                is_deleted=remote_group['is_deleted'],
                abc_service_id=remote_group['abc_service_id'],
            )

        update_users(group, local_users, remote_group['users'], threshold)

        if need_save:
            group_to_save.append(group)
            group_to_save_count += 1

        if group_to_save_count > BATCH_SIZE:
            group_to_save_batches.append(group_to_save)
            group_to_save = []
            group_to_save_count = 0

    for group in local_groups.values():
        # если какие-то группы совсем опустели, то staff нам ничего про них не вернул,
        # чистим такие группы от пользователей
        update_users(group, local_users, set(), threshold)
        group.is_deleted = True
        group.save()

    group_to_save_batches.append(group_to_save)
    for batch in group_to_save_batches:
        with transaction.atomic():
            for group in batch:
                group.save()
    log.info('Serviceroles synchronization with staff is complete')


class SyncGroupsTask(CrtBaseTask):
    task_type = TASK_TYPE.SYNC_GROUPS

    def run(
        self,
        force=False,
        threshold=REMOVE_USERS_FROM_SCOPE_THRESHOLD,
        inconsistency_skip_threshold=SKIP_INCONSISTENT_GROUPS_THRESHOLD,
        **kwargs
    ):
        if force:
            threshold = 1.0
            inconsistency_skip_threshold = 1.0
        remote_groups = get_service_roles(inconsistency_skip_threshold)
        update_groups(remote_groups, threshold)
