import datetime
import logging
from typing import Dict, Iterable, List
import pytz
import yenv

from django.conf import settings
from django.db import transaction
from django.utils import timezone
from django.utils import six

from ids.registry import registry

from intranet.crt.constants import AFFILIATION, TASK_TYPE, INCONSISTENCY_TYPE
from intranet.crt.tasks.base import CrtBaseTask
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.http import CrtSession
from intranet.crt.monitorings.models import add_inconsistencies

HIRING_ROBOT_OFFICE = 144
USERS_BATCH_SIZE = 250
NAME_ATTRIBUTES = ('first_name', 'last_name', 'first_name_ru', 'last_name_ru')


def get_staff_repository(resource_type: str):
    return registry.get_repository(
        'staff', resource_type,
        user_agent=settings.CRT_IDS_USER_AGENT,
        oauth_token=settings.CRT_OAUTH_TOKEN,
        timeout=settings.CRT_STAFF_TIMEOUT,
    )


yenv.type = getattr(settings, 'CRT_IDS_OVERRIDE_ENV_TYPE', yenv.type)
staff_repo = get_staff_repository('person')
groups_repo = get_staff_repository('group')
log = logging.getLogger(__name__)


def date_to_utc(str_date: str) -> datetime.datetime:
    date = datetime.datetime.strptime(str_date, '%Y-%m-%d')
    return pytz.utc.localize(date)


def in_hiring_user_is_robot(person):
    """ В наниматоре роботом является тот сотрудник, который входит в определенный офис
    """
    return person['office'] == HIRING_ROBOT_OFFICE


def get_offers():
    # 30 - заявка подтверждена руководителем, 40 - подготовка рабочего места
    response = CrtSession().set_oauth().get(
        settings.CRT_NEWHIRE_API_URL,
    )
    return response.json()


def get_department_urls_to_affiliations_dict(department_urls: Iterable[str]) -> Dict[str, str]:
    def get_group_affiliation(group_name: str, ancestors: List[str]) -> str:
        if (
            group_name == 'yandex' or
            len(ancestors) > 0 and ancestors[0]['url'] == AFFILIATION.YANDEX
        ):
            return AFFILIATION.YANDEX
        return AFFILIATION.EXTERNAL

    response = groups_repo.getiter(lookup={
        '_fields': ','.join((
            'url',
            'ancestors.url',
        )),
        'url': ','.join(set(department_urls))
    })
    groups = {
        group['url']: get_group_affiliation(group['url'], group['ancestors'])
        for group in response
    }
    return groups


def get_in_hiring_users():
    newhire_offers = get_offers()
    department_urls = (offer['department_url'] for offer in newhire_offers)
    department_url_to_affiliation = get_department_urls_to_affiliations_dict(department_urls)
    offers = {}
    ignored_incosistent_users = []
    now = timezone.now()
    for offer in newhire_offers:
        if not offer['username']:
            continue

        join_at = timezone.utc.localize(datetime.datetime.strptime(offer['join_at'], '%Y-%m-%d'))
        # join_at – предполагаемая рекрутёрами дата выхода человека на работу
        # если она слишком в прошлом, то человек, кажется, слишком долго принимал приглашение, и его не нужно уже
        # синхронизировать. Если же эта дата слишком в будущем, то, значит, человек выйдёт ещё не скоро,
        # пока что незачем ему готовить оборудование и заводить в CRT
        if abs(now - join_at) > datetime.timedelta(days=settings.CRT_NEWHIRE_FRESHNESS_DAYS):
            continue

        username = offer['username'].strip().lower()
        department_url = offer['department_url']

        if department_url not in department_url_to_affiliation:
            ignored_incosistent_users.append(f'{username}: department {department_url} was not found')
            continue

        offers[username] = {
            'username': username,
            'is_active': False,
            'first_name': offer['first_name_en'] or '',
            'last_name': offer['last_name_en'] or '',
            'first_name_ru': offer['first_name'] or '',
            'last_name_ru': offer['last_name'] or '',
            'email': '{}@yandex-team.ru'.format(username),
            'date_joined': date_to_utc(offer['join_at']),
            'in_hiring': True,
            'is_robot': in_hiring_user_is_robot(offer),
            'country': None,
            'city': None,
            'unit': None,
            'lang_ui': 'ru',
            'affiliation': department_url_to_affiliation[department_url],
        }
    add_inconsistencies(ignored_incosistent_users, INCONSISTENCY_TYPE.USER)
    return offers


def dict_getpath(data, path):
    current_data = data
    for component in path.split('.'):
        current_data = current_data.get(component) or {}

    return current_data or None


def get_unit(person):
    department_groups = person['department_group'].get('ancestors', [])
    department_groups.append(person['department_group'])

    direction_url = None
    for group in reversed(department_groups):
        slug = dict_getpath(group, 'department.kind.slug')
        if slug == 'direction':
            direction_url = dict_getpath(group, 'department.url')

    if direction_url is None:
        return 'TOP'
    else:
        return direction_url.split('_')[-1].capitalize()


def get_staff_users():
    lookup = {
        '_sort': 'id',
        '_limit': 500,
        '_fields': ','.join((
            'id',
            'login',
            'official.is_dismissed',
            'official.join_at',
            'official.is_robot',
            'official.affiliation',
            'location.office.city.name.en',
            'location.office.city.country.code',
            'department_group.ancestors.department.kind.slug',
            'department_group.ancestors.department.url',
            'department_group.department.kind.slug',
            'department_group.department.url',
            'name.first.en',
            'name.last.en',
            'name.first.ru',
            'name.last.ru',
            'work_email',
            'language.ui',
            'robot_owners.person.login',
        )),
    }

    result = {}
    query_template = 'id > %d'
    last_id = 0
    while True:
        lookup['_query'] = query_template % last_id
        persons = list(staff_repo.getiter(lookup=lookup).first_page)
        if len(persons) == 0:
            break
        for person in persons:
            result[person['login']] = {
                'username': person['login'],
                'is_active': not person['official']['is_dismissed'],
                'first_name': person['name']['first']['en'],
                'last_name': person['name']['last']['en'],
                'first_name_ru': person['name']['first']['ru'],
                'last_name_ru': person['name']['last']['ru'],
                'email': person['work_email'],
                'date_joined': date_to_utc(person['official']['join_at']),
                'in_hiring': False,
                'is_robot': person['official']['is_robot'],
                'country': person['location']['office']['city']['country']['code'],
                'city': person['location']['office']['city']['name']['en'],
                'unit': get_unit(person),
                'lang_ui': person['language']['ui'],
                'robot_owners': person.get('robot_owners', []),
                'affiliation': person['official']['affiliation'],
            }
        last_id = person['id']
    return result


def update_fullname(user):
    if user.first_name and user.last_name:
        user.full_name = '{} {}'.format(user.first_name, user.last_name)
    else:
        user.full_name = ''
    if user.first_name_ru and user.last_name_ru:
        user.full_name_ru = '{} {}'.format(user.first_name_ru, user.last_name_ru)
    else:
        user.full_name_ru = ''


def save_users(remote_users):
    local_users = {user.username: user for user in CrtUser.objects.all()}
    users_to_save = []
    users_to_save_batches = []
    users_to_save_count = 0
    for remote_user in remote_users.values():
        if remote_user['username'] in local_users:
            user = local_users[remote_user['username']]
        else:
            user = CrtUser(username=remote_user['username'])
            users_to_save.append(user)
            users_to_save_count += 1

        fullname_is_up_to_date = True
        for attr, remote_value in remote_user.items():
            if attr == 'robot_owners':
                continue
            local_value = getattr(user, attr)
            if local_value != remote_value:
                setattr(user, attr, remote_value)
                users_to_save.append(user)
                users_to_save_count += 1
                if attr in NAME_ATTRIBUTES:
                    fullname_is_up_to_date = False

        if not fullname_is_up_to_date:
            update_fullname(user)

        if users_to_save_count > USERS_BATCH_SIZE:
            users_to_save_batches.append(users_to_save)
            users_to_save = []
            users_to_save_count = 0

    users_to_save_batches.append(users_to_save)
    for batch in users_to_save_batches:
        with transaction.atomic():
            for user in batch:
                user.save()

    not_provided = local_users.keys() - remote_users.keys()
    expired_inhiring = {username for username in not_provided if local_users[username].in_hiring}
    usernames = ', '.join(sorted(expired_inhiring))
    log.info('Following users are not in_hiring anymore: %s', usernames)
    CrtUser.objects.filter(username__in=expired_inhiring).update(in_hiring=False)


def save_robot_responsibilities(remote_users):
    staff_ownings_data = {
        username: {owner['person']['login'] for owner in remote_users[username].get('robot_owners', [])}
        for username in remote_users
    }

    robots_usernames = {username for username in remote_users if remote_users[username]['is_robot']}

    crt_ownings_data = {
        user.username: {
            'user_object': user,
            'robot_owners': {owner.username for owner in user.responsibles.all()}
        }
        for user in CrtUser.objects.filter(username__in=robots_usernames).prefetch_related('responsibles')
    }
    for username in robots_usernames:

        staff_robot_owners = staff_ownings_data[username]
        crt_robot_owners = crt_ownings_data[username]['robot_owners']

        added = staff_robot_owners - crt_robot_owners
        if added:
            to_add = CrtUser.objects.filter(username__in=added)
            crt_ownings_data[username]['user_object'].responsibles.add(*to_add)

            log.info(
                'New responsibles added to robot %s: %s' % (
                    staff_ownings_data[username],
                    ', '.join(user.username for user in to_add)
                )
            )

        removed = crt_robot_owners - staff_robot_owners
        if removed:
            to_remove = CrtUser.objects.filter(username__in=removed)
            crt_ownings_data[username]['user_object'].responsibles.remove(*to_remove)
            log.info(
                'Responsibles removed from robot %s: %s' % (
                    staff_ownings_data[username],
                    ', '.join(user.username for user in to_remove)
                )
            )


class SyncUsersTask(CrtBaseTask):
    task_type = TASK_TYPE.SYNC_USERS

    def run(self, **kwargs):
        # Нанимаемых синкаем первыми, чтобы не терять их во время вывода на staff
        in_hiring_users = get_in_hiring_users()
        remote_users = get_staff_users()

        for username, userdata in remote_users.items():
            if username in in_hiring_users:
                userdata['in_hiring'] = True
                userdata['affiliation'] = in_hiring_users[username]['affiliation']

        new_usernames = set(in_hiring_users.keys()) - set(remote_users.keys())
        for username in new_usernames:
            remote_users[username] = in_hiring_users[username]

        save_users(remote_users)
        save_robot_responsibilities(remote_users)
