import constance

from django.db import models
from django.contrib.auth.models import AbstractUser
from django.utils.translation import ugettext_lazy as _

from intranet.crt.constants import CRT_GROUP_TYPE, AFFILIATION, CERT_TYPE
from intranet.crt.users.managers import CrtUserManager, GroupManager
from intranet.crt.core.utils import old_get_inums_and_models
from intranet.crt.utils.constance import get_values_set_from_str


class CrtUser(AbstractUser):
    in_hiring = models.BooleanField(default=False)
    is_robot = models.BooleanField(default=False)
    country = models.CharField(max_length=7, null=True, blank=True)
    city = models.CharField(max_length=31, null=True, blank=True)
    unit = models.CharField(max_length=255, null=True, blank=True)
    lang_ui = models.CharField(max_length=10, blank=True, default='ru')
    affiliation = models.CharField(max_length=8, db_index=True, choices=AFFILIATION.CHOICES,
                                   null=False, blank=False, default=AFFILIATION.OTHER)

    first_name = models.CharField(_('first name'), max_length=60, blank=True)
    first_name_ru = models.CharField(_('first name(ru)'), max_length=60, blank=True)
    last_name = models.CharField(_('last name'), max_length=60, blank=True)
    last_name_ru = models.CharField(_('last name(ru)'), max_length=60, blank=True)
    full_name = models.CharField(_('full name'), max_length=121, blank=True, db_index=True)
    full_name_ru = models.CharField(_('full name(ru)'), max_length=121, blank=True, db_index=True)

    ninja_certs_quota = models.IntegerField(null=True, blank=True)

    staff_groups = models.ManyToManyField('CrtGroup', related_name='users')

    responsibles = models.ManyToManyField('self', symmetrical=False, related_name='robots')

    objects = CrtUserManager()

    def get_robot_owner(self):
        raise NotImplementedError()
        # TODO

    def is_external_employee(self):
        return self.affiliation == AFFILIATION.EXTERNAL and self.is_robot is False

    def has_active_ld_certificates(self):
        return self.certificates.filter(
            type__name__in=CERT_TYPE.INTRANET_ACCESS_TYPES,
            common_name=f'{self.username}@ld.yandex.ru',
        ).issued().exists()

    def is_pdas_whitelisted(self):
        return self.username in get_values_set_from_str(constance.config.PDAS_WHITELIST)

    def has_bot_hardware(self):
        return bool(old_get_inums_and_models(self.username))


class CrtGroup(models.Model):
    type = models.CharField(max_length=30, choices=CRT_GROUP_TYPE.choices())
    external_id = models.IntegerField(db_index=True)
    name = models.TextField(default='', db_index=True)
    url = models.TextField()
    role_scope = models.TextField(default='', null=True, db_index=True)
    is_deleted = models.BooleanField()
    abc_service = models.ForeignKey(
        'django_abc_data.AbcService',
        null=True,
        blank=True,
        default=None,
        on_delete=models.SET_NULL,
        related_name='crt_group',
    )

    objects = GroupManager()

    def __repr__(self):
        return 'Group {0}'.format(self.name.encode('utf-8'))
