import abc
import sys
import logging
import traceback

from django.core.management.base import BaseCommand as DjangoBaseCommand

from ylog.context import log_context

from intranet.crt.utils.lock import locked_context
from intranet.crt.utils.random import generate_task_id


class BaseCommand(DjangoBaseCommand, metaclass=abc.ABCMeta):
    USE_LOCK = True

    @abc.abstractmethod
    def _handle(self, *args, **options):
        pass

    @property
    def command_name(self):
        return self.__class__.__module__.rsplit('.')[-1]

    def execute_command(self, *args, **options):
        logging.info('Start command {}'.format(self.command_name))

        try:
            self._handle(*args, **options)
        except:
            traceback.print_exc()
            logging.exception('Unhandled exception')
            sys.exit(1)

        logging.info('Complete command {}'.format(self.command_name))

    def handle(self, *args, **options):
        self.command_id = generate_task_id()
        with log_context(command_id=self.command_id):
            if not self.USE_LOCK:
                self.execute_command(*args, **options)
                return

            with locked_context(self.command_name) as locked:
                if not locked:
                    return

                self.execute_command(*args, **options)
