import re
import csv
import datetime

import attr
from django.conf import settings
from django.utils import timezone
from rest_framework import status

from intranet.crt.exceptions import CrtTimestampError
from intranet.crt.utils.http import CrtSession
from intranet.crt.utils.time import utc_to_moscow, get_date_range

oid_pattern = re.compile(r'([\d.]+)(?:\s+|$).*', re.UNICODE)


def oid_converter(value):
    match = oid_pattern.match(value)
    if match is None:
        return value

    return match.group(1)


def string_converter(value):
    return value if value != 'EMPTY' else None


def date_converter(value):
    date_string = string_converter(value)
    if date_string is None:
        return None

    return timezone.pytz.timezone(settings.TIME_ZONE).localize(
        datetime.datetime.strptime(date_string, '%d.%m.%Y %H:%M')
    )


@attr.s
class CertInfo(object):
    request_id = attr.ib()
    oid = attr.ib(converter=oid_converter)
    common_name = attr.ib()
    not_after = attr.ib(converter=date_converter)
    revoke_date = attr.ib(converter=date_converter)
    user_principal_name = attr.ib(converter=string_converter)
    email = attr.ib(converter=string_converter)
    serial_number = attr.ib(converter=str.upper)


class InternalCaExportLoader(object):
    def __init__(self):
        self.session = CrtSession().set_basic_robot_auth()

    def load_data_from_url(self, url):
        response = self.session.get(url)

        if response.status_code != status.HTTP_200_OK:
            raise CrtTimestampError(
                'Invalid status code %s from %s. Content: %s',
                response.status_code, settings.INTERNAL_CA, response.content
            )

        return response.text.splitlines()

    def load_full_data(self):
        return self.load_data_from_url(settings.CRT_INTERNAL_CA_EXPORT_FULL_URL)

    def load_diff_data(self, from_date):
        now = timezone.now()
        to_date = utc_to_moscow(now)
        if from_date is None:
            from_date = now - datetime.timedelta(days=1)
            from_date = utc_to_moscow(from_date)

        remote_data = []
        for date in get_date_range(from_date, to_date):
            date = date.strftime('%d%m%Y')
            url = settings.CRT_INTERNAL_CA_EXPORT_DIFF_URL_PATTERN.format(date)
            next_remote_data = self.load_data_from_url(url)
            remote_data.extend(next_remote_data)

        return remote_data

    def load(self, full=True, from_date=None):
        if full:
            return self.load_full_data()

        return self.load_diff_data(from_date)


class InternalCaExport(object):
    def __init__(self, certs_info):
        self.certs_info = certs_info

    @classmethod
    def load(cls, full=True, from_date=None):
        loader = InternalCaExportLoader()
        if full:
            remote_data = loader.load_full_data()
        else:
            remote_data = loader.load_diff_data(from_date)

        data = csv.reader(remote_data)
        certs_info = [CertInfo(*line) for line in data]
        certs_info = {info.serial_number: info for info in certs_info}

        return cls(certs_info)
