import abc

import attr
import pytz
from cryptography import x509
from cryptography.hazmat.backends import default_backend

from django.conf import settings
from django.utils import timezone
from django.utils.functional import cached_property
from rest_framework import status

from intranet.crt.constants import CERT_STATUS
from intranet.crt.exceptions import CrtError
from intranet.crt.utils.http import CrtSession
from intranet.crt.utils.ssl import serial_number_converter


def status_converter(extensions):
    extensions_len = len(extensions)

    if extensions_len > 1:
        raise CrtError('Invalid certificate in CRL. Found {} extensions'.format(extensions_len))

    if extensions_len == 0:
        return CERT_STATUS.REVOKED

    if extensions[0].value.reason == x509.ReasonFlags.certificate_hold:
        return CERT_STATUS.HOLD

    return CERT_STATUS.REVOKED


@attr.s
class CrlCertInfo(object):
    serial_number = attr.ib(converter=serial_number_converter)
    status = attr.ib(converter=status_converter)
    revoke_date = attr.ib(converter=pytz.utc.localize)


class BaseCaCrl(object, metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def crl_url(self):
        pass

    @abc.abstractmethod
    def crl_loader(self):
        pass

    def __init__(self, crl):
        self.crl = crl

    @classmethod
    def load(cls):
        response = CrtSession().get(cls.crl_url)

        if response.status_code != status.HTTP_200_OK:
            raise CrtError('Invalid status code {code} from {url}. Content: {content}'.format(
                code=response.status_code,
                url=cls.crl_url,
                content=response.content,
            ))

        crl = cls.crl_loader(response.content)

        return cls(crl)

    @property
    def last_update(self):
        return timezone.utc.localize(self.crl.last_update)

    @property
    def next_update(self):
        return timezone.utc.localize(self.crl.next_update)

    @cached_property
    def certs(self):
        certificates = {}
        for revoked_cert in self.crl:
            cert_info = CrlCertInfo(
                serial_number=revoked_cert.serial_number,
                revoke_date=revoked_cert.revocation_date,
                status=revoked_cert.extensions,
            )
            certificates[cert_info.serial_number] = cert_info
        return certificates

    @cached_property
    def hold_certs(self):
        return {cert.serial_number for cert in self.certs.values() if cert.status == CERT_STATUS.HOLD}

    @cached_property
    def revoked_certs(self):
        return {cert.serial_number for cert in self.certs.values() if cert.status == CERT_STATUS.REVOKED}


class InternalCaCrl(BaseCaCrl):
    crl_url = settings.INTERNAL_CA_CRL
    crl_loader = default_backend().load_der_x509_crl
