import json
import pwd
import pipes
import shutil
import os.path

import pexpect
from django.conf import settings

from intranet.crt.exceptions import CrtTimestampError


class CvsClient(object):
    CVS_COMMAND = '/usr/bin/cvs'
    UP_COMMAND = 'up'
    COMMIT_COMMAND = 'commit -m {}'
    CHECKOUT_COMMAND = '-d:ext:tree.yandex.ru:/opt/CVSROOT/ checkout "{repo}"'.format(
        repo=settings.CRT_CVS_REPO_NAME,
    )

    def __init__(self, check_user=True):
        """ https://st.yandex-team.ru/CERTOR-726 """
        if check_user:
            self._check_user()

    @classmethod
    def _execute(cls, command, cwd=settings.CRT_CVS_FILE_DIR):
        cvs_command = ' '.join([cls.CVS_COMMAND, command])
        cvs = pexpect.spawn(cvs_command, cwd=cwd)
        try:
            cvs.expect(pexpect.EOF, timeout=settings.CRT_CVS_TIMEOUT)
        except pexpect.TIMEOUT:
            raise CrtTimestampError(
                '{command} exit with timeout {timeout}. Output: {output}'
                .format(command=cvs_command, timeout=settings.CRT_CVS_TIMEOUT, output=cvs.before)
            )
        if cvs.exitstatus:
            raise CrtTimestampError(
                '{command} exit with exit code {code}: {details}'
                .format(command=cvs_command, code=cvs.exitstatus, details=cvs.before)
            )

    @classmethod
    def _check_user(cls):
        """ Нельзя запускать checkout не от www-data, иначе cron-таски будут падать на rmtree """
        uid = os.getuid()
        username = pwd.getpwuid(uid).pw_name
        if username != settings.CRT_OS_USERNAME:
            raise CrtTimestampError('Can not run cvs from user {}'.format(username))

    @classmethod
    def is_checkouted(cls):
        return os.path.isdir(settings.CRT_CVS_FILE_DIR)

    def checkout(self, force=False):
        if force:
            shutil.rmtree(settings.CRT_CVS_FILE_DIR)

        if self.is_checkouted():
            return False

        self._execute(self.CHECKOUT_COMMAND, cwd=settings.CRT_CVS_REPO_DIR)

        return True

    def up(self, force=False):
        if self.checkout(force=force):
            return

        self._execute(self.UP_COMMAND)

    def commit(self, commit_message=None):
        if commit_message is None:
            commit_message = 'CRT auto updated NOC tags'

        commit_message = pipes.quote(commit_message)

        command = self.COMMIT_COMMAND.format(commit_message)
        self._execute(command)


class NocCvsClient(CvsClient):
    def _get_file(self, filename, up=False):
        if up:
            self.up()

        with open(filename, 'r') as cvs_file:
            data = json.load(cvs_file)

        return data

    def get_old_certs_with_types(self, up=False):
        return self._get_file(settings.CRT_CVS_OLD_DATA_FILE, up=up)

    def get_diff(self, up=False):
        return self._get_file(settings.CRT_CVS_DIFF_FILE, up=up)
