import re
import socket

import dns.resolver
from django.utils.encoding import force_text

CERTUM_VALIDATION_CODE_RE = re.compile(r'^[a-f0-9]+$', re.IGNORECASE)
GLOBALSIGN_VALIDATION_CODE_RE = re.compile(r'^_globalsign-domain-verification=[a-zA-Z0-9_-]+$')


def get_family_ips(host, family):
    try:
        return list({ret[4][0] for ret in socket.getaddrinfo(host, None, family)})
    except (socket.gaierror, UnicodeError):
        return list()


def get_host_ips(host):
    ips_v4 = get_family_ips(host, socket.AF_INET)
    ips_v6 = get_family_ips(host, socket.AF_INET6)
    return ips_v4 + ips_v6


def get_ptrs_by_ip(ip):
    reverse_name = dns.reversename.from_address(ip)
    ptr_names = []
    try:
        ptr_names = dns.resolver.query(reverse_name, dns.rdatatype.PTR)
    # UnicodeDecodeError падает, если в ptr_names есть запретные символы
    except (dns.exception.DNSException, UnicodeDecodeError):
        pass
    texts = {ptr.to_text().strip('.') for ptr in ptr_names}
    return texts


def double_resolve(fqdn):
    """Resolve FQDN to a set of IP addresses, then back-resolve each IP address to PTR record,
    return a set of FQDNs in PTR records"""
    all_ptrs = set()
    ips = get_host_ips(fqdn)
    for ip in ips:
        ptrs = get_ptrs_by_ip(ip)
        all_ptrs |= ptrs
    return all_ptrs


def get_host_txt_codes(host, name_servers, globalsign_validation=False, certum_validation=True):
    if globalsign_validation:
        validation_code_re = GLOBALSIGN_VALIDATION_CODE_RE
    elif certum_validation:
        validation_code_re = CERTUM_VALIDATION_CODE_RE
    else:
        raise RuntimeError("You should select unknown validation type for DNS TXT records")

    name_servers_ips = []
    for name_server in name_servers:
        name_servers_ips += get_host_ips(name_server.strip())

    resolver = dns.resolver.Resolver()
    resolver.nameservers = name_servers_ips
    dns_answer = resolver.query(host, dns.rdatatype.TXT)

    codes = set()
    for data in dns_answer:
        for code in data.strings:
            code = force_text(code)
            if validation_code_re.match(code) is not None:
                codes.add(code)
    return codes


def get_name_servers(host):
    host = host.rstrip('.') + '.'
    try:
        response = dns.resolver.query(host, dns.rdatatype.NS)
    except dns.resolver.NXDOMAIN:
        return set()

    return {name_server.to_text().strip('.') for name_server in response}


def name_servers_are_managed(name_servers, managed_servers):
    managed = name_servers & managed_servers
    if managed:
        return True
    all_aliases = set()
    for name_server in name_servers:
        aliases = double_resolve(name_server)
        all_aliases |= aliases
    return bool(all_aliases & managed_servers)


def get_aliases(host):
    aliases = [host]

    try:
        response = dns.resolver.query(host, dns.rdatatype.CNAME)
    except dns.resolver.NoAnswer:
        return aliases

    aliases.extend([cname.to_text().strip('.') for cname in response])
    return aliases
